#pragma once

#include <yxiva/core/binary_protocol.h>
#include <yxiva/core/types.h>
#include <ymod_httpclient/cluster_client.h>
#include <optional>

namespace yxiva {

struct rproxy_rate_limit_settings
{
    using errc = binary_protocol::error_code;

    uint64_t limit = 0;
    uint64_t recovery_rate = 0;
    uint64_t recovery_interval_ms = 0;
    errc response_code = errc::service_unavailable;

    void parse_ptree(const yplatform::ptree& ptree)
    {
        limit = ptree.get<uint64_t>("limit");
        recovery_rate = ptree.get<uint64_t>("recovery_rate");
        auto recovery_interval = ptree.get<time_duration>("recovery_interval");
        if (recovery_interval < milliseconds(1))
            throw std::runtime_error("recovery interval must be greater than 1ms");
        recovery_interval_ms = duration_cast<milliseconds>(recovery_interval).count();
        response_code = static_cast<errc>(ptree.get<int>("response_code"));
    }
};

struct rproxy_settings
{
    struct gate_settings
    {
        yhttp::cluster_client::settings http;
        std::optional<rproxy_rate_limit_settings> rate_limit;
    };

    std::map<string, gate_settings> backends;
    bool allow_url_override = false;
    size_t max_frame_size = 10 * 1024 * 1024;
    time_duration stats_ttl = seconds(30);

    bool enabled_for_any(const std::vector<string>& services) const
    {
        for (auto& service : services)
        {
            if (backends.count(service))
            {
                return true;
            }
        }
        return false;
    }

    void parse_ptree(const yplatform::ptree& ptree)
    {
        for ([[maybe_unused]] auto& [ignored, gate_ptree] :
             boost::make_iterator_range(ptree.equal_range("backends")))
        {
            auto service = gate_ptree.get<string>("service");
            backends[service].http.parse_ptree(gate_ptree.get_child("http"));
            if (auto&& rate_limit_conf = gate_ptree.get_child_optional("rate_limit"))
            {
                rproxy_rate_limit_settings st;
                st.parse_ptree(*rate_limit_conf);
                backends[service].rate_limit = st;
            }
        }
        allow_url_override = ptree.get("allow_url_override", allow_url_override);
        max_frame_size = ptree.get("max_frame_size", max_frame_size);
        stats_ttl = ptree.get("stats_ttl", stats_ttl);
    }
};

inline rproxy_settings create_rproxy_settings(const yplatform::ptree& ptree)
{
    rproxy_settings ret;
    ret.parse_ptree(ptree);
    return ret;
}

}
