#include "impl.h"

#include <yplatform/find.h>
#include <yplatform/util/safe_call.h>
#include <yplatform/hash/sha1.h>
#include <boost/asio/yield.hpp>
#include <algorithm>
#include <stdexcept>

namespace yxiva {
namespace {
static const auto ACTION_CREATED = "created";
static const auto ACTION_UPDATED = "updated";
static const auto ACTION_REVOKED = "revoked";
static const auto ACTION_RESTORED = "restored";

string token_gen(
    const string& salt,
    const token_properties& data,
    ymod_xconf::config_environment env)
{
    string id;
    yplatform::sstream(id) << data.item_name() << ymod_xconf::get_environment_name(env) << salt
                           << std::to_string(time(nullptr));
    return yplatform::sha1(id);
}

string token_gen(const send_token_properties& data, ymod_xconf::config_environment env)
{
    static const string salt = "_xivatesthubtoken";
    return token_gen(salt, data, env);
}

string token_gen(const listen_token_properties& data, ymod_xconf::config_environment env)
{
    static const string salt = "_xivatesthubtoken_squire";
    return token_gen(salt, data, env);
}

operation::result result_from_error_code(const ymod_xconf::error_code& ec)
{
    return ec ? ec.message() : operation::success;
}
}

namespace p = std::placeholders;

service_manager_impl::service_manager_impl() : xconf_update_active_(false), max_revision_(0)
{
}

service_manager_impl::service_manager_impl(
    const ymod_xconf::local_conf_storage_ptr& xconf,
    boost::asio::io_service& io,
    const string& environment)
    : xconf_(xconf)
    , environment_(environment)
    , update_strand_(std::make_shared<boost::asio::io_service::strand>(io))
    , xconf_update_active_(false)
    , max_revision_(0)
{
}

void service_manager_impl::init(const yplatform::ptree& config)
{
    xconf_ = yplatform::find_module<ymod_xconf::local_conf_storage>(
        config.get<string>("xconf_module", ""));
    update_strand_ = std::make_shared<boost::asio::io_service::strand>(
        *yplatform::find_reactor(config.get("reactor", "global"))->io());

    environment_ = config.get<string>("environment");
    if (boost::algorithm::iequals(environment_, "any"))
    {
        environment_.clear();
    }
    if (environment_.size())
    {
        auto env = ymod_xconf::resolve_environment(environment_);
        if (env == ymod_xconf::config_environment::UNKNOWN)
        {
            throw std::runtime_error("unknown environment");
        }
        xconf_->subscribe_updates(
            ymod_xconf::config_type::ANY,
            env,
            update_strand_->wrap(
                std::bind(&service_manager_impl::load_configs, shared_from(this), p::_1)));
    }
    else
    {
        xconf_->subscribe_updates(
            ymod_xconf::config_type::ANY,
            update_strand_->wrap(
                std::bind(&service_manager_impl::load_configs, shared_from(this), p::_1)));
    }
}

std::shared_ptr<const service_data> service_manager_impl::find_service_by_send_token(
    const string& token) const
{
    scoped_lock guard(lock_);
    auto it = services_by_send_token_.find(token);
    return it == services_by_send_token_.end() ? nullptr : it->second;
}

std::shared_ptr<const service_data> service_manager_impl::find_service_by_listen_token(
    const string& token) const
{
    scoped_lock guard(lock_);
    auto it = services_by_listen_token_.find(token);
    return it == services_by_listen_token_.end() ? nullptr : it->second;
}

std::shared_ptr<const service_data> service_manager_impl::find_service_by_name(
    const string& name) const
{
    scoped_lock guard(lock_);
    auto it = services_by_name_.find(name);
    return it == services_by_name_.end() ? nullptr : it->second;
}

std::shared_ptr<const application_config> service_manager_impl::find_app(
    const string& platform,
    const string& name) const
{
    scoped_lock guard(lock_);
    // Find name of service that owns app.
    auto key = std::tie(platform::resolve_alias(platform).name, name); // gcm_compatibility
    auto owner_it = app_owners_by_name_.find(key);
    if (owner_it == app_owners_by_name_.end()) return nullptr;
    // Get owner.
    auto service_it = services_by_name_.find(owner_it->second);
    if (service_it == services_by_name_.end()) return nullptr;
    // Get the actual app.
    auto it = service_it->second->apps.find(key);
    // Return pointer to app, lifetime is shared with service object.
    return it == service_it->second->apps.end() ?
        nullptr :
        std::shared_ptr<const application_config>{ service_it->second, &it->second };
}

void service_manager_impl::find_services_by_owner(
    const task_context_ptr& ctx,
    const string& owner,
    const services_callback_t& cb)
{
    // Get the latest configuration first.
    schedule_xconf_update(
        [ctx, cb, owner, this, self = shared_from(this)](const operation::result& res) {
            if (ctx->is_cancelled())
            {
                return;
            }
            safe_call(
                ctx,
                "service manager handler",
                cb,
                res,
                res ? find_services_by_owner_impl(owner) : services_type{});
        });
}

void service_manager_impl::find_services_by_owners(
    const task_context_ptr& ctx,
    std::shared_ptr<const std::vector<string>> owners,
    const services_callback_t& cb)
{
    // Get the latest configuration first.
    schedule_xconf_update([ctx, cb, owners = std::move(owners), this, self = shared_from(this)](
                              const operation::result& res) {
        if (ctx->is_cancelled())
        {
            return;
        }
        safe_call(
            ctx,
            "service manager handler",
            cb,
            res,
            res ? find_services_by_owners_impl(owners) : services_type{});
    });
}

void service_manager_impl::get_all_services(
    const task_context_ptr& ctx,
    const services_callback_t& cb)
{
    // Get the latest configuration first.
    schedule_xconf_update([ctx, cb, this, self = shared_from(this)](const operation::result& res) {
        if (ctx->is_cancelled())
        {
            return;
        }
        safe_call(
            ctx,
            "service manager handler",
            cb,
            res,
            res ? get_all_services_impl() : services_type{});
    });
}

void service_manager_impl::update_service_properties(
    task_context_ptr ctx,
    const service_properties& data,
    const update_callback_t& cb)
{
    auto replace_properties = [data](auto&&) { return std::move(data); };
    update_service_properties(ctx, data.name, std::move(replace_properties), cb);
}

void service_manager_impl::update_service_properties(
    task_context_ptr ctx,
    const string& service_name,
    const service_properties_transform& transform,
    const update_callback_t& cb)
{
    // Owner is checked by xconf, but still need to check service exists.
    schedule_xconf_update([cb, ctx, service_name, transform, this, self = shared_from(this)](
                              const operation::result& res) mutable {
        if (ctx->is_cancelled())
        {
            return;
        }
        if (!res)
        {
            safe_call(ctx, "service manager handler", cb, res);
            return;
        }
        auto service = find_service_by_name(service_name);
        if (!service)
        {
            safe_call(ctx, "service manager handler", cb, "service doesn't exist");
            return;
        }
        // xconf_owner_deprecated
        xconf_->put(
            ymod_xconf::config_type::SERVICE,
            service_name,
            service->properties.deprecated_owner,
            "",
            max_revision_,
            boost::make_shared<string>(pack(transform(service->properties))),
            [ctx, cb, self](const ymod_xconf::error_code& ec, ymod_xconf::revision_t) {
                safe_call(ctx, "service manager handler", cb, result_from_error_code(ec));
            },
            ctx);
    });
}

void service_manager_impl::update_send_token(
    task_context_ptr ctx,
    const string& env_str,
    const send_token_properties& data,
    const string& service_owner,
    const token_update_callback_t& cb)
{
    if (!allowed(env_str))
    {
        return safe_call(
            ctx, "service manager handler", cb, "environment is not allowed", string());
    }
    auto env = ymod_xconf::resolve_environment(env_str);
    if (env == ymod_xconf::config_environment::UNKNOWN)
    {
        return safe_call(ctx, "service manager handler", cb, "unknown environment", string());
    }
    schedule_xconf_update([data, service_owner, cb, ctx, env, this, self = shared_from(this)](
                              const operation::result& res) {
        update_send_token_impl(ctx, env, res, data, service_owner, cb);
    });
}

void service_manager_impl::update_listen_token(
    task_context_ptr ctx,
    const string& env_str,
    const listen_token_properties& data,
    const string& service_owner,
    const token_update_callback_t& cb)
{
    if (!allowed(env_str))
    {
        return safe_call(
            ctx, "service manager handler", cb, "environment is not allowed", string());
    }
    auto env = ymod_xconf::resolve_environment(env_str);
    if (env == ymod_xconf::config_environment::UNKNOWN)
    {
        return safe_call(ctx, "service manager handler", cb, "unknown environment", string());
    }
    schedule_xconf_update([data, service_owner, cb, ctx, env, this, self = shared_from(this)](
                              const operation::result& res) mutable {
        update_listen_token_impl(ctx, env, res, data, service_owner, cb);
    });
}

void service_manager_impl::create_service(
    task_context_ptr ctx,
    const service_properties& data,
    const create_service_callback_t& cb)
{
    send_token_properties default_stoken_props;
    default_stoken_props.service = data.name;
    default_stoken_props.name = "default";
    default_stoken_props.revoked = false;

    listen_token_properties default_ltoken_props;
    default_ltoken_props.service = data.name;
    default_ltoken_props.name = "default";
    default_ltoken_props.revoked = false;
    default_ltoken_props.client = data.name;

    auto task = std::make_shared<detail::create_service_task>();
    task->service_draft = data;
    task->ctx = std::move(ctx);
    task->cb = cb;
    task->env_id = 0;
    // Create token drafs.
    for (size_t i = 0; i < static_cast<size_t>(ymod_xconf::config_environment::COUNT); ++i)
    {
        auto env = static_cast<ymod_xconf::config_environment>(i);
        task->stoken_drafts.push_back(
            { env, token_gen(default_stoken_props, env), default_stoken_props });
        task->ltoken_drafts.push_back(
            { env, token_gen(default_ltoken_props, env), default_ltoken_props });
    }

    schedule_xconf_update([task, this, self = shared_from(this)](const operation::result& res) {
        create_service_impl(res, task, boost::asio::coroutine{});
    });
}

void service_manager_impl::update_app(
    task_context_ptr ctx,
    const application_config& data,
    const string& service_owner,
    const update_callback_t& cb)
{
    if (data.xiva_service.empty())
    {
        return safe_call(ctx, "service manager handler", cb, "must be owned by service");
    }
    schedule_xconf_update(
        [data, service_owner, cb, ctx, this, self = shared_from(this)](
            const operation::result& res) { update_app_impl(ctx, res, data, service_owner, cb); });
}

void service_manager_impl::schedule_xconf_update(update_callback_t&& cb)
{
    scoped_lock guard(lock_);
    update_handlers_.emplace_back(std::move(cb));
    if (!xconf_update_active_)
    {
        xconf_update_active_ = true;
        if (environment_.size())
        {
            xconf_->list(
                ymod_xconf::config_type::ANY,
                ymod_xconf::resolve_environment(environment_),
                max_revision_,
                update_strand_->wrap(std::bind(
                    &service_manager_impl::handle_xconf_update, shared_from(this), p::_1, p::_2)),
                boost::make_shared<task_context>());
        }
        else
        {
            xconf_->list(
                ymod_xconf::config_type::ANY,
                max_revision_,
                update_strand_->wrap(std::bind(
                    &service_manager_impl::handle_xconf_update, shared_from(this), p::_1, p::_2)),
                boost::make_shared<task_context>());
        }
    }
}

void service_manager_impl::handle_xconf_update(
    const ymod_xconf::error_code& err,
    ymod_xconf::conf_list_ptr confs)
{
    std::vector<update_callback_t> handlers;
    {
        scoped_lock guard(lock_);
        xconf_update_active_ = false;
        std::swap(update_handlers_, handlers);
    }
    if (!err)
    {
        load_configs(confs);
    }
    auto res_ptr = std::make_shared<operation::result>(result_from_error_code(err));
    for (auto& handler : handlers)
    {
        update_strand_->get_io_service().post([h = std::move(handler), res_ptr]() { h(*res_ptr); });
    }
}

void service_manager_impl::load_configs(ymod_xconf::conf_list_ptr confs)
{
    size_t current_rev = max_revision_;
    // Assign early for xconf requests from outside of strand.
    max_revision_ = confs->max_revision;

    for (auto& item : confs->items)
    {
        if (item.revision <= current_rev)
        {
            continue;
        }
        try
        {
            switch (item.type)
            {
            case ymod_xconf::config_type::SERVICE:
                process_item<service_properties>(item);
                break;
            case ymod_xconf::config_type::SEND_TOKEN:
                process_item<send_token_properties>(item);
                break;
            case ymod_xconf::config_type::LISTEN_TOKEN:
                process_item<listen_token_properties>(item);
                break;
            case ymod_xconf::config_type::MOBILE:
                process_item<application_config>(item);
                break;
            default:; // Other types are not supported
            }
        }
        catch (const std::exception& ex)
        {
            YLOG_L(error) << "service manager failed to unpack configuration \"" << item.name
                          << "\" of type " << ymod_xconf::get_type_name(item.type) << ": "
                          << ex.what();
        }
    }
}

template <typename DataType>
void service_manager_impl::process_item(const ymod_xconf::item& item)
{
    DataType data;
    unpack(item.configuration, data);
    apply_entity(item, std::move(data));
}

void service_manager_impl::update_service(
    const std::shared_ptr<const service_data>& old_service,
    const std::shared_ptr<service_data>& new_service)
{
    scoped_lock guard(lock_);
    services_by_name_[new_service->properties.name] = new_service;
    // Create or replace services_by_owner_ record. Owner can be empty if non-service
    // record is currently applied (before the corresponding service record).
    auto owner = new_service->properties.owner();
    if (owner.size())
    {
        auto range = services_by_owner_.equal_range(owner);
        auto it = std::find_if(
            range.first,
            range.second,
            [&old_service](const std::pair<string, std::shared_ptr<service_data>>& rec) {
                return rec.second == old_service;
            });
        if (it == range.second)
        {
            services_by_owner_.insert(std::make_pair(owner, new_service));
        }
        else
        {
            it->second = new_service;
        }
    }
    // Update app owners.
    for (auto& app : new_service->apps)
    {
        app_owners_by_name_[app.first] = new_service->properties.name;
    }
    // As service can have same tokens in different environments,
    // some tokens may be revoked in part of environments. To maintain
    // service_by_*_token lists, token maps will be scanned twice,
    // once to remove revoked records and again to add valid ones,
    // as not to have record valid for one env erased because of another env.
    for (auto& send_token : new_service->send_tokens)
    {
        if (new_service->properties.revoked || send_token.second.revoked)
        {
            services_by_send_token_.erase(send_token.first.token);
        }
    }
    for (auto& listen_token : new_service->listen_tokens)
    {
        if (new_service->properties.revoked || listen_token.second.revoked)
        {
            services_by_listen_token_.erase(listen_token.first.token);
        }
    }
    for (auto& send_token : new_service->send_tokens)
    {
        if (!new_service->properties.revoked && !send_token.second.revoked)
        {
            services_by_send_token_[send_token.first.token] = new_service;
        }
    }
    for (auto& listen_token : new_service->listen_tokens)
    {
        if (!new_service->properties.revoked && !listen_token.second.revoked)
        {
            services_by_listen_token_[listen_token.first.token] = new_service;
        }
    }
}

void service_manager_impl::apply_entity(const ymod_xconf::item& item, service_properties data)
{
    data.deprecated_owner = item.owner; // xconf_owner_deprecated
    auto old_service = find_service_by_name(data.name);
    // Copy old service object or create new.
    auto new_service = old_service ? std::make_shared<service_data>(*old_service) :
                                     std::make_shared<service_data>();
    std::swap(new_service->properties, data);
    update_service(old_service, new_service);

    const char* action;
    // Deduce action from previous and new state.
    if (old_service)
    {
        if (old_service->properties.revoked == new_service->properties.revoked)
        {
            action = ACTION_UPDATED;
        }
        else
        {
            action = new_service->properties.revoked ? ACTION_REVOKED : ACTION_RESTORED;
        }
    }
    else
    {
        action = new_service->properties.revoked ? ACTION_REVOKED : ACTION_CREATED;
    }
    YLOG_L(info) << "service " << new_service->properties.name << " is " << action
                 << " on revision " << item.revision;
}

void service_manager_impl::apply_entity(const ymod_xconf::item& item, send_token_properties data)
{
    auto old_service = find_service_by_name(data.service);
    // Copy old service object or create new.
    std::shared_ptr<service_data> new_service = old_service ?
        std::make_shared<service_data>(*old_service) :
        std::make_shared<service_data>();

    if (!old_service)
    {
        // If service record is created by token item, set service name!
        new_service->properties.name = data.service;
    }
    auto& token_props =
        (new_service->send_tokens[{ item.token, item.environment }] = std::move(data));
    update_service(old_service, new_service);

    const char* action;
    // Deduce action from previous and new state.
    if (old_service)
    {
        auto it = old_service->send_tokens.find({ item.token, item.environment });
        if (it == old_service->send_tokens.end())
        {
            action = token_props.revoked ? ACTION_REVOKED : ACTION_CREATED;
        }
        else
        {
            if (token_props.revoked == it->second.revoked)
            {
                action = ACTION_UPDATED;
            }
            else
            {
                action = token_props.revoked ? ACTION_REVOKED : ACTION_RESTORED;
            }
        }
    }
    else
    {
        action = token_props.revoked ? ACTION_REVOKED : ACTION_CREATED;
    }
    YLOG_L(info) << "send token " << item.name << " is " << action << " on revision "
                 << item.revision << " for environment " << item.environment;
}

void service_manager_impl::apply_entity(const ymod_xconf::item& item, listen_token_properties data)
{
    auto old_service = find_service_by_name(data.service);
    // Copy old service object or create new.
    std::shared_ptr<service_data> new_service = old_service ?
        std::make_shared<service_data>(*old_service) :
        std::make_shared<service_data>();

    if (!old_service)
    {
        // If service record is created by token item, set service name!
        new_service->properties.name = data.service;
    }
    auto& token_props =
        (new_service->listen_tokens[{ item.token, item.environment }] = std::move(data));
    update_service(old_service, new_service);

    const char* action;
    // Deduce action from previous and new state.
    if (old_service)
    {
        auto it = old_service->listen_tokens.find({ item.token, item.environment });
        if (it == old_service->listen_tokens.end())
        {
            action = token_props.revoked ? ACTION_REVOKED : ACTION_CREATED;
        }
        else
        {
            if (token_props.revoked == it->second.revoked)
            {
                action = ACTION_UPDATED;
            }
            else
            {
                action = token_props.revoked ? ACTION_REVOKED : ACTION_RESTORED;
            }
        }
    }
    else
    {
        action = token_props.revoked ? ACTION_REVOKED : ACTION_CREATED;
    }
    YLOG_L(info) << "listen token " << item.name << " is " << action << " on revision "
                 << item.revision << " for environment " << item.environment;
}

void service_manager_impl::apply_entity(const ymod_xconf::item& item, application_config data)
{
    data.platform = platform::resolve_alias(data.platform).name; // gcm_compatibility
    if (data.xiva_service.empty())
    {
        YLOG_L(info) << "can't apply revision " << item.revision
                     << ": user ownership for apps is deprecated, owner is " << data.owner_id();
        return;
    }

    auto old_service = find_service_by_name(data.xiva_service);
    // Copy old service object or create new.
    std::shared_ptr<service_data> new_service = old_service ?
        std::make_shared<service_data>(*old_service) :
        std::make_shared<service_data>();
    if (!old_service)
    {
        // If service record is created by mobile item, set service name!
        new_service->properties.name = data.xiva_service;
    }

    // Store app regardless of presence of secret key to enable update and revert.
    new_service->apps[std::tie(data.platform, data.app_name)] = std::move(data);
    update_service(old_service, new_service);
}

services_type service_manager_impl::find_services_by_owner_impl(const string& owner)
{
    size_t service_count;
    {
        scoped_lock guard(lock_);
        service_count = services_by_owner_.count(owner);
    }
    // Try to reserve adequate amount of space outside of lock.
    services_type services;
    services.reserve(service_count);

    scoped_lock guard(lock_);
    auto range = services_by_owner_.equal_range(owner);
    for (auto it = range.first; it != range.second; ++it)
    {
        services.push_back(it->second);
    }
    return services;
}

services_type service_manager_impl::find_services_by_owners_impl(
    std::shared_ptr<const std::vector<string>> owners)
{
    size_t service_count = 0;
    {
        scoped_lock guard(lock_);
        for (auto& owner : *owners)
        {
            service_count += services_by_owner_.count(owner);
        }
    }
    // Try to reserve adequate amount of space outside of lock.
    services_type services;
    services.reserve(service_count);

    scoped_lock guard(lock_);
    for (auto& owner : *owners)
    {
        auto range = services_by_owner_.equal_range(owner);
        for (auto it = range.first; it != range.second; ++it)
        {
            services.push_back(it->second);
        }
    }
    return services;
}

services_type service_manager_impl::get_all_services_impl()
{
    size_t service_count;
    {
        scoped_lock guard(lock_);
        service_count = services_by_name_.size();
    }
    // Try to reserve adequate amount of space outside of lock.
    services_type services;
    services.reserve(service_count);

    scoped_lock guard(lock_);
    for (auto& service : services_by_name_)
    {
        services.push_back(service.second);
    }
    return services;
}

void service_manager_impl::update_send_token_impl(
    task_context_ptr ctx,
    ymod_xconf::config_environment env,
    const operation::result& res,
    const send_token_properties& data,
    const string& service_owner,
    const token_update_callback_t& cb)
{
    if (ctx->is_cancelled())
    {
        return;
    }
    if (!res)
    {
        safe_call(ctx, "service manager handler", cb, res, string());
        return;
    }

    auto service = find_service_by_name(data.service);
    if (!service)
    {
        safe_call(ctx, "service manager handler", cb, "service not found", string());
    }
    else if (service->properties.owner() != service_owner)
    {
        safe_call(ctx, "service manager handler", cb, "service owner mismatch", string());
    }
    else
    {
        // Try to find this token by name.
        auto& tokens = service->send_tokens;
        auto env_name = ymod_xconf::get_environment_name(env);
        auto token_it = std::find_if(
            tokens.begin(),
            tokens.end(),
            [&data, &env_name](const std::pair<service_data::key_t, send_token_properties>& rec) {
                return data.name == rec.second.name && env_name == rec.first.env;
            });
        // Update existing token or generate new.
        string token =
            (token_it == tokens.end()) ? token_gen(data, env) : token_it->first.source_token;
        auto packed = boost::make_shared<string>(pack(data));
        xconf_->put(
            ymod_xconf::config_type::SEND_TOKEN,
            env,
            data.item_name(),
            data.service,
            token,
            max_revision_,
            packed,
            [ctx, token, cb, self = shared_from(this)](
                const ymod_xconf::error_code& ec, ymod_xconf::revision_t) {
                safe_call(ctx, "service manager handler", cb, result_from_error_code(ec), token);
            },
            ctx);
    }
}

void service_manager_impl::update_listen_token_impl(
    task_context_ptr ctx,
    ymod_xconf::config_environment env,
    const operation::result& res,
    const listen_token_properties& data,
    const string& service_owner,
    const token_update_callback_t& cb)
{
    if (ctx->is_cancelled())
    {
        return;
    }
    if (!res)
    {
        safe_call(ctx, "service manager handler", cb, res, string());
        return;
    }

    auto service = find_service_by_name(data.service);
    if (!service)
    {
        safe_call(ctx, "service manager handler", cb, "service not found", string());
    }
    else if (service->properties.owner() != service_owner)
    {
        safe_call(ctx, "service manager handler", cb, "service owner mismatch", string());
    }
    else
    {
        // Try to find this token by name.
        auto& tokens = service->listen_tokens;
        auto env_name = ymod_xconf::get_environment_name(env);
        auto token_it = std::find_if(
            tokens.begin(),
            tokens.end(),
            [&data, &env_name](const std::pair<service_data::key_t, listen_token_properties>& rec) {
                return data.name == rec.second.name && env_name == rec.first.env;
            });
        // Update existing token or generate new.
        string token =
            (token_it == tokens.end()) ? token_gen(data, env) : token_it->first.source_token;
        auto packed = boost::make_shared<string>(pack(data));
        xconf_->put(
            ymod_xconf::config_type::LISTEN_TOKEN,
            env,
            data.item_name(),
            data.service,
            token,
            max_revision_,
            packed,
            [ctx, token, cb, self = shared_from(this)](
                const ymod_xconf::error_code& ec, ymod_xconf::revision_t) {
                safe_call(ctx, "service manager handler", cb, result_from_error_code(ec), token);
            },
            ctx);
    }
}

void service_manager_impl::create_service_impl(
    const operation::result& res,
    std::shared_ptr<detail::create_service_task> task,
    boost::asio::coroutine coro)
{
    string error;
    auto& service_properties = task->service_draft;
    auto& ctx = task->ctx;

    if (!res)
    {
        // Don't reenter if previous action failed
        error = res.error_reason;
    }
    else
    {
        try
        {
            reenter(coro)
            {
                if (find_service_by_name(service_properties.name))
                {
                    error = "service already exists";
                    yield break;
                }

                yield xconf_->put(
                    ymod_xconf::config_type::SERVICE,
                    service_properties.name,
                    service_properties.owner(),
                    "",
                    max_revision_,
                    boost::make_shared<string>(pack(service_properties)),
                    [ctx, task, coro, this, self = shared_from(this)](
                        const ymod_xconf::error_code& ec, ymod_xconf::revision_t) {
                        create_service_impl(result_from_error_code(ec), task, coro);
                    },
                    ctx);

                // Create service object if service record was written successfully.
                task->service_result = std::make_shared<service_data>();
                task->service_result->properties = service_properties;

                for (; task->env_id < static_cast<size_t>(ymod_xconf::config_environment::COUNT);
                     ++task->env_id)
                {
                    yield xconf_->put(
                        ymod_xconf::config_type::SEND_TOKEN,
                        task->stoken_draft().env,
                        task->stoken_draft().props.item_name(),
                        task->stoken_draft().props.service,
                        task->stoken_draft().token,
                        max_revision_,
                        boost::make_shared<string>(pack(task->stoken_draft().props)),
                        [ctx, task, coro, this, self = shared_from(this)](
                            const ymod_xconf::error_code& ec, ymod_xconf::revision_t) {
                            create_service_impl(result_from_error_code(ec), task, coro);
                        },
                        ctx);
                    // If send token was written successfully, add it to service
                    task->service_result->send_tokens[{
                        task->stoken_draft().token,
                        ymod_xconf::get_environment_name(task->stoken_draft().env) }] =
                        task->stoken_draft().props;

                    yield xconf_->put(
                        ymod_xconf::config_type::LISTEN_TOKEN,
                        task->ltoken_draft().env,
                        task->ltoken_draft().props.item_name(),
                        task->ltoken_draft().props.service,
                        task->ltoken_draft().token,
                        max_revision_,
                        boost::make_shared<string>(pack(task->ltoken_draft().props)),
                        [ctx, task, coro, this, self = shared_from(this)](
                            const ymod_xconf::error_code& ec, ymod_xconf::revision_t) {
                            create_service_impl(result_from_error_code(ec), task, coro);
                        },
                        ctx);
                    // If listen token was written successfully, add it to service
                    task->service_result->listen_tokens[{
                        task->ltoken_draft().token,
                        ymod_xconf::get_environment_name(task->ltoken_draft().env) }] =
                        task->ltoken_draft().props;
                }
            }
        }
        catch (const std::exception& ex)
        {
            error = ex.what();
        }
    }
    // Return result if everything is done or if coro didn't run because of error.
    if (coro.is_complete() || !res)
    {
        safe_call(
            task->ctx,
            "service manager handler",
            task->cb,
            error.empty() ? operation::success : error,
            task->service_result);
    }
}

void service_manager_impl::update_app_impl(
    task_context_ptr ctx,
    const operation::result& res,
    const application_config& data,
    const string& service_owner,
    const update_callback_t& cb)
{
    if (ctx->is_cancelled())
    {
        return;
    }
    if (!res)
    {
        safe_call(ctx, "service manager handler", cb, res);
        return;
    }

    auto service = find_service_by_name(data.xiva_service);
    if (!service)
    {
        return safe_call(ctx, "service manager handler", cb, "service not found");
    }
    else if (service->properties.owner() != service_owner)
    {
        return safe_call(ctx, "service manager handler", cb, "service owner mismatch");
    }

    auto packed = boost::make_shared<string>(pack(data));
    xconf_->put(
        ymod_xconf::config_type::MOBILE,
        data.name(),
        data.owner_id(),
        "",
        max_revision_,
        packed,
        [ctx, cb, self = shared_from(this)](
            const ymod_xconf::error_code& ec, ymod_xconf::revision_t) {
            safe_call(ctx, "service manager handler", cb, result_from_error_code(ec));
        },
        ctx);
}

}

#include <boost/asio/unyield.hpp>

#include <yplatform/module_registration.h>
REGISTER_MODULE(yxiva::service_manager_impl)
