#pragma once

#include "interface.h"
#include <ymod_xconf/local_conf_storage.h>
#include <yplatform/module.h>
#include <yplatform/reactor.h>
#include <yplatform/spinlock.h>
#include <atomic>
#include <functional>
#include <memory>
#include <unordered_map>
#include <vector>

namespace yxiva {

namespace detail {
template <typename TokenProps>
struct token_draft
{
    ymod_xconf::config_environment env;
    string token;
    TokenProps props;
};

struct create_service_task
{
    service_properties service_draft;
    std::shared_ptr<service_data> service_result;
    task_context_ptr ctx;
    create_service_callback_t cb;
    std::vector<token_draft<send_token_properties>> stoken_drafts;
    std::vector<token_draft<listen_token_properties>> ltoken_drafts;
    size_t env_id;

    const token_draft<send_token_properties>& stoken_draft() const
    {
        return stoken_drafts[env_id];
    }
    const token_draft<listen_token_properties>& ltoken_draft() const
    {
        return ltoken_drafts[env_id];
    }
};
}

class service_manager_impl
    : public yplatform::module
    , public service_manager
{
    using spinlock = yplatform::spinlock;
    using scoped_lock = std::lock_guard<spinlock>;

public:
    service_manager_impl();
    service_manager_impl(
        const ymod_xconf::local_conf_storage_ptr&,
        boost::asio::io_service&,
        const string&);

    void init(const yplatform::ptree& config);

    std::shared_ptr<const service_data> find_service_by_send_token(const string& token) const;
    std::shared_ptr<const service_data> find_service_by_listen_token(const string& token) const;
    std::shared_ptr<const service_data> find_service_by_name(const string& name) const;
    std::shared_ptr<const application_config> find_app(const string& platform, const string& name)
        const;
    void find_services_by_owner(
        const task_context_ptr& ctx,
        const string& owner,
        const services_callback_t& cb);
    // In case of lookup for multiple owners, prevents multiple xconf::list() requests.
    void find_services_by_owners(
        const task_context_ptr& ctx,
        std::shared_ptr<const std::vector<string>> owners,
        const services_callback_t& cb);
    // Gets all services (needed for admin users).
    void get_all_services(const task_context_ptr& ctx, const services_callback_t& cb);
    const string& environment() const
    {
        return environment_;
    }

    void update_service_properties(
        task_context_ptr ctx,
        const service_properties& data,
        const update_callback_t& cb);
    void update_service_properties(
        task_context_ptr ctx,
        const string& service_name,
        const service_properties_transform& transform,
        const update_callback_t& cb);
    void update_send_token(
        task_context_ptr ctx,
        const string& env_str,
        const send_token_properties& data,
        const string& service_owner,
        const token_update_callback_t& cb);
    void update_listen_token(
        task_context_ptr ctx,
        const string& env_str,
        const listen_token_properties& data,
        const string& service_owner,
        const token_update_callback_t& cb);
    void create_service(
        task_context_ptr ctx,
        const service_properties& data,
        const create_service_callback_t& cb);
    void update_app(
        task_context_ptr ctx,
        const application_config& data,
        const string& service_owner,
        const update_callback_t& cb);

protected:
    // Must be executed in strand.
    void load_configs(ymod_xconf::conf_list_ptr confs);

    void schedule_xconf_update(update_callback_t&& cb);

private:
    // Must be executed in strand.
    void handle_xconf_update(const ymod_xconf::error_code& err, ymod_xconf::conf_list_ptr confs);

    // Must be executed in strand.
    template <typename DataType>
    void process_item(const ymod_xconf::item& item);

    // Create or update service record.
    void update_service(
        const std::shared_ptr<const service_data>& old_service,
        const std::shared_ptr<service_data>& new_service);

    // Apply data of given type from the item.
    // Must be executed in strand.
    void apply_entity(const ymod_xconf::item& item, service_properties data);
    void apply_entity(const ymod_xconf::item& item, send_token_properties data);
    void apply_entity(const ymod_xconf::item& item, listen_token_properties data);
    void apply_entity(const ymod_xconf::item& item, application_config data);

    services_type find_services_by_owner_impl(const string& owner);
    services_type find_services_by_owners_impl(std::shared_ptr<const std::vector<string>> owners);
    services_type get_all_services_impl();

    void update_send_token_impl(
        task_context_ptr ctx,
        ymod_xconf::config_environment env,
        const operation::result& res,
        const send_token_properties& data,
        const string& service_owner,
        const token_update_callback_t& cb);
    void update_listen_token_impl(
        task_context_ptr ctx,
        ymod_xconf::config_environment env,
        const operation::result& res,
        const listen_token_properties& data,
        const string& service_owner,
        const token_update_callback_t& cb);
    void create_service_impl(
        const operation::result& res,
        std::shared_ptr<detail::create_service_task> task,
        boost::asio::coroutine coro);
    void update_app_impl(
        task_context_ptr ctx,
        const operation::result& res,
        const application_config& data,
        const string& service_owner,
        const update_callback_t& cb);

    bool allowed(const string& env)
    {
        return environment_.empty() || environment_ == env;
    }

    ymod_xconf::local_conf_storage_ptr xconf_;
    string environment_;
    std::shared_ptr<boost::asio::io_service::strand> update_strand_;
    bool xconf_update_active_;
    std::vector<update_callback_t> update_handlers_;
    mutable spinlock lock_;
    // Needs to be atomic for reads from outside of strand.
    std::atomic_size_t max_revision_;

    std::unordered_map<string, std::shared_ptr<service_data>> services_by_name_;
    std::unordered_multimap<string, std::shared_ptr<service_data>> services_by_owner_;
    std::unordered_map<string, std::shared_ptr<service_data>> services_by_send_token_;
    std::unordered_map<string, std::shared_ptr<service_data>> services_by_listen_token_;
    // Key is {platform, app_name}.
    std::map<std::tuple<string, string>, string> app_owners_by_name_;
};

}
