#pragma once

#include "common.h"
#include "web/find_deps.h"
#include "web/extract_requests.h"
#include <yxiva/core/filter.h>
#include <yxiva/core/split.h>
#include <yxiva/core/operation_result.h>

namespace yxiva { namespace web { namespace api { namespace params {

namespace detail {

template <typename Stream>
inline void param_find(Stream& stream, const string& name, string& dst)
{
    auto it = stream->request()->url.param_find(name);
    if (it != stream->request()->url.params.end()) dst = it->second;
}

template <typename Stream, typename ValidateFunc>
bool simple_param(
    Stream& stream,
    const string& name,
    ValidateFunc&& validator,
    string& dst,
    bool optional = false)
{
    string raw;
    param_find(stream, name, raw);
    if (raw.empty() && !optional)
    {
        send_bad_request(stream, "missing " + name + " argument");
        return false;
    }
    if (validate_param(raw, validator))
    {
        dst = raw;
        return true;
    }
    else
    {
        send_bad_request(stream, "bad characters in " + name + " argument");
        return false;
    }
}

}

#define YXIVA_API_SIMPLE_PARAM_DEF(param, name, arg_name, validator, optional)                     \
    struct param                                                                                   \
    {                                                                                              \
        template <typename Stream, typename Arguments>                                             \
        static bool parse(Stream& stream, Arguments& args)                                         \
        {                                                                                          \
            return detail::simple_param(stream, #name, (validator), args->arg_name, optional);     \
        }                                                                                          \
    };

#define YXIVA_API_SIMPLE_PARAM(param, validator)                                                   \
    YXIVA_API_SIMPLE_PARAM_DEF(param, param, param, validator, false)                              \
    YXIVA_API_SIMPLE_PARAM_DEF(optional_##param, param, param, validator, true)
#define YXIVA_API_EXT_PARAM(param, name, arg_name, validator)                                      \
    YXIVA_API_SIMPLE_PARAM_DEF(param, name, arg_name, validator, false)                            \
    YXIVA_API_SIMPLE_PARAM_DEF(optional_##param, name, arg_name, validator, true)

YXIVA_API_SIMPLE_PARAM(uid, isalnum)
YXIVA_API_SIMPLE_PARAM(client, isalnum_ext)
YXIVA_API_SIMPLE_PARAM(session, isalnum_ext)
YXIVA_API_SIMPLE_PARAM(event, isalnum_ext)
YXIVA_API_SIMPLE_PARAM(service, isalnum_ext)
YXIVA_API_EXT_PARAM(service_raw, service, service, [](char) { return true; })
YXIVA_API_EXT_PARAM(tags_list, tags, tags, tag_list_char)
YXIVA_API_SIMPLE_PARAM(local_id, isdigit)
YXIVA_API_SIMPLE_PARAM(ttl, isdigit)
YXIVA_API_EXT_PARAM(mobile_application_name, app_name, app_name, isalnum_ext)
YXIVA_API_EXT_PARAM(mobile_platform, platform, platform, isalnum_ext)
YXIVA_API_EXT_PARAM(mobile_push_token, push_token, push_token, [](char) { return true; })
YXIVA_API_EXT_PARAM(oauth_token, oauth_token, oauth_token, [](char) { return true; })
YXIVA_API_SIMPLE_PARAM(subscription_id, isalnum_ext)

struct xiva_token
{
    template <typename Stream, typename Arguments>
    static bool parse(Stream& stream, Arguments& args)
    {
        string field_name;
        if (stream->request()->url.param_count("stoken"))
        {
            field_name = "stoken";
        }
        else if (stream->request()->url.param_count("ctoken"))
        {
            field_name = "ctoken";
        }
        else
        {
            field_name = "token";
        }
        return detail::simple_param(stream, field_name, isalnum, args->xiva_token);
    }
};

struct oauth_token_or_cookie
{
    template <typename Stream, typename Arguments>
    static bool parse(Stream& stream, Arguments& args)
    {
        if (stream->request()->url.param_count("oauth_token"))
        {
            return oauth_token::parse(stream, args);
        }
        else
        {
            args->cookie = stream->request()->headers["cookie"];
            return true;
        }
    }
};

struct uid_as_list
{
    template <typename Stream, typename Arguments>
    static bool parse(Stream& stream, Arguments& args)
    {
        string raw;
        detail::param_find(stream, "uid", raw);
        if (raw.empty())
        {
            send_bad_request(stream, "missing uid argument");
            return false;
        }

        args->uids_list = utils::split(raw, ",");
        if (args->uids_list.empty())
        {
            send_bad_request(stream, "empty uids list");
            return false;
        }

        return true;
    }
};

struct service_list_with_tags
{
    template <typename Stream, typename Arguments>
    static bool parse(Stream& stream, Arguments& args)
    {
        string raw;
        detail::param_find(stream, "service", raw);
        if (raw.empty())
        {
            send_bad_request(stream, "missing service argument");
            return false;
        }
        channels_set channels;
        try
        {
            channels = compose_channels(raw, "", "", user_info());
        }
        catch (const std::exception& /*e*/)
        {
            // Response message taken as is from /v1/subscribe
            send_bad_request(stream, "invalid service argument");
            return false;
        }
        if (channels.empty())
        {
            send_bad_request(stream, "empty service list");
            return false;
        }
        args->services.reserve(channels.size());
        for (auto& channel : channels)
        {
            args->services.push_back(service_with_filter{ channel.service, channel.filter });
        }
        return true;
    }
};

struct callback
{
    template <typename Stream, typename Arguments>
    static bool parse(Stream& stream, Arguments& args)
    {
        // @todo: smartly validate callback
        return detail::simple_param(
            stream, "callback", [](char) { return true; }, args->callback);
    }
};

template <typename Arg1, typename... Args>
struct parser
{
    template <typename Stream, typename Arguments>
    static bool fill(Stream& stream, Arguments& args)
    {
        return parser<Arg1>::fill(stream, args) && parser<Args...>::fill(stream, args);
    }
};

template <typename Arg>
struct parser<Arg>
{
    template <typename Stream, typename Arguments>
    static bool fill(Stream& stream, Arguments& args)
    {
        return Arg::parse(stream, args);
    }
};

}}}}