#include "apns_queue_repeat.h"

#include <yxiva/core/platforms.h>

namespace yxiva { namespace web { namespace api2 {

bool hub_request_failed(const boost::system::error_code& err, yhttp::response response);
void log_hub_request_failed(
    task_context_ptr ctx,
    const boost::system::error_code& err,
    yhttp::response response);
int repeated_count_from_response(const boost::system::error_code& err, yhttp::response response);

apns_queue_repeat::apns_queue_repeat(settings_ptr settings) : settings(settings)
{
}

void apns_queue_repeat::operator()(
    http_stream_ptr stream,
    settings_ptr /*settings*/,
    const service_authorization& /*auth*/,
    const string& /*service*/,
    const string& app_name,
    const string& device_id,
    local_id_t position,
    local_id_t count)
{
    namespace p = std::placeholders;

    if (!settings->api.apns_queue.enabled)
    {
        return stream->result(http_codes::bad_request, "apns queue is disabled");
    }
    if (count > settings->api.apns_queue.max_repeat_count)
    {
        return stream->result(http_codes::bad_request, "count is too big");
    }
    auto queue_id = apns_queue_id(device_id, app_name);
    // Transform last seen position into start position.
    find_hubrpc()->async_get(
        stream->ctx(),
        queue_id,
        "/repeat_messages",
        { { "service", settings->api.apns_queue.service },
          { "uid", queue_id },
          { "position", position - count },
          { "count", count } },
        [stream](const boost::system::error_code& err, yhttp::response response) {
            try
            {
                if (hub_request_failed(err, response))
                {
                    log_hub_request_failed(stream->ctx(), err, response);
                }
                json_value resp;
                auto&& result = resp["result"];
                result["repeated_count"] = repeated_count_from_response(err, response);
                stream->set_code(http_codes::ok);
                stream->set_content_type("application/json");
                stream->result_body(resp.stringify());
            }
            catch (const std::exception& ex)
            {
                LERR_(stream->ctx())
                    << "apns_queue_repeat exception=\"" << ex.what() << "\" body=" << response.body;
            }
        });
}

bool hub_request_failed(const boost::system::error_code& err, yhttp::response response)
{
    return err || response.status != http_codes::ok;
}

void log_hub_request_failed(
    task_context_ptr ctx,
    const boost::system::error_code& err,
    yhttp::response response)
{
    LERR_(ctx) << "hub error=" << err.message() << " status=" << response.status;
}

int repeated_count_from_response(const boost::system::error_code& err, yhttp::response response)
{
    if (hub_request_failed(err, response))
    {
        return 0;
    }
    try
    {
        return stoi(response.body);
    }
    catch (const std::invalid_argument&)
    {
        return 0;
    }
}

}}}
