#pragma once

#include <yxiva/core/batch_key.h>
#include <yxiva/core/notify_status_to_json.h>
#include <yxiva/core/gid.h>
#include <yxiva/core/message.h>
#include <yxiva/core/notify_status.h>
#include <boost/noncopyable.hpp>
#include <numeric>

namespace yxiva { namespace web { namespace api2 {

namespace {
static const string JSON_FIELD_CODE("code");
static const string JSON_FIELD_BODY("body");
static const string JSON_FIELD_ID("id");
static const string JSON_FIELD_DUPLICATE("duplicate");
static const string JSON_FIELD_RESULTS("results");
}

struct batch_codes_acc
{
    void add(int code, size_t value)
    {
        switch (code)
        {
        case 200:
            codes_200 += value;
            break;
        case 204:
            codes_204 += value;
            break;
        case 205:
            codes_205 += value;
            break;
        case 429:
            codes_429 += value;
            break;
        case 500:
            codes_500 += value;
            break;
        case 502:
            codes_502 += value;
            break;
        case 503:
            codes_503 += value;
            break;
        case 504:
            codes_504 += value;
            break;
        }
        int code_class = code / 100;
        switch (code_class)
        {
        case 2:
            codes_2xx += value;
            break;
        case 4:
            codes_4xx += value;
            break;
        case 5:
            codes_5xx += value;
            break;
        }
    }

    static void dump(ymod_webserver::context& ctx, const batch_codes_acc& acc)
    {
        ctx.custom_log_data["batch_200"] = std::to_string(acc.codes_200);
        ctx.custom_log_data["batch_204"] = std::to_string(acc.codes_204);
        ctx.custom_log_data["batch_205"] = std::to_string(acc.codes_205);
        ctx.custom_log_data["batch_429"] = std::to_string(acc.codes_429);
        ctx.custom_log_data["batch_500"] = std::to_string(acc.codes_500);
        ctx.custom_log_data["batch_502"] = std::to_string(acc.codes_502);
        ctx.custom_log_data["batch_503"] = std::to_string(acc.codes_503);
        ctx.custom_log_data["batch_504"] = std::to_string(acc.codes_504);
        ctx.custom_log_data["batch_2xx"] = std::to_string(acc.codes_2xx);
        ctx.custom_log_data["batch_4xx"] = std::to_string(acc.codes_4xx);
        ctx.custom_log_data["batch_5xx"] = std::to_string(acc.codes_5xx);
    }

    // special
    std::atomic<size_t> codes_200 = { 0 };
    std::atomic<size_t> codes_204 = { 0 };
    std::atomic<size_t> codes_205 = { 0 };
    std::atomic<size_t> codes_429 = { 0 };
    std::atomic<size_t> codes_500 = { 0 };
    std::atomic<size_t> codes_502 = { 0 };
    std::atomic<size_t> codes_503 = { 0 };
    std::atomic<size_t> codes_504 = { 0 };
    // total
    std::atomic<size_t> codes_2xx = { 0 };
    std::atomic<size_t> codes_4xx = { 0 };
    std::atomic<size_t> codes_5xx = { 0 };
};

struct batch_context : public boost::noncopyable
{
    http_stream_ptr stream;
    yplatform::net::streamable_ptr stream_chunks;
    ::yxiva::message message;
    batch_keys keys;

    // These indices track the order in which keys
    // will be packed to chunks and sent to xiva hub servers.
    std::vector<size_t> send_index;
    std::vector<json_value> answer;
    batch_codes_acc codes;

    batch_context(
        const http_stream_ptr& stream,
        ::yxiva::message&& tmp_message,
        batch_keys&& tmp_keys)
        : stream(stream), message(std::move(tmp_message)), keys(std::move(tmp_keys))
    {
        // Fills array with null values. Later we will replace them by
        // reports from hub(s).
        answer.resize(keys.size());

        build_send_index();

        stream->set_code(http_codes::ok);
        stream->set_content_type("application/json");
        stream->add_header("TransitID", message.transit_id);
        stream_chunks = stream->result_chunked();
    }

    ~batch_context()
    {
        assert(answer.size());

        batch_codes_acc::dump(*stream->request()->context, codes);

        { // scope for ostream
            auto ostream = stream_chunks->client_stream();
            for (auto it = answer.begin(); it != answer.end(); ++it)
            {
                if (it->is_null())
                {
                    *it = to_json(notify_status(500, "internal timeout"));
                }
                if (it != answer.begin())
                {
                    ostream << "," << it->stringify();
                }
                else
                {
                    ostream << "{\"results\":[" << it->stringify();
                }
            }
            ostream << "]}";
        }

        stream_chunks.reset();
    }

    void report(size_t index_start, size_t index_end, const std::vector<notify_status>& results)
    {
        for (auto& status : results)
        {
            size_t offset = index_start + status.seq;
            if (offset >= index_end)
            {
                YLOG_CTX_GLOBAL(stream->ctx(), error)
                    << "incorrect report index " << status.seq << " out of range [" << index_start
                    << "," << index_end << "]";
                continue;
            }

            // Extend the answer null->object->array.
            auto& value = answer[send_index[offset]];
            if (value.is_null())
            {
                value = to_json(status);
            }
            else if (value.is_array())
            {
                value.push_back(to_json(status));
            }
            else
            {
                json_value old_value;
                old_value.swap(value);
                value = json_value();
                value.push_back(old_value);
                value.push_back(to_json(status));
            }

            codes.add(status.code, 1);
        }
    }

    void report_error(size_t index_start, size_t index_end, const notify_status& status)
    {
        for (size_t i = index_start; i < index_end; ++i)
        {
            answer[send_index[i]] = to_json(status);
        }
        codes.add(status.code, index_end - index_start);
    }

private:
    // Actually sorts only indices of original keys.
    std::vector<size_t> sort_keys_by_uid()
    {
        auto index_uid = std::vector<size_t>(keys.size());
        std::iota(index_uid.begin(), index_uid.end(), 0);
        std::sort(index_uid.begin(), index_uid.end(), [this](size_t i, size_t j) -> bool {
            return std::tie(keys[i].uid, keys[i].subscription_id) <
                std::tie(keys[j].uid, keys[j].subscription_id);
        });
        return index_uid;
    }

    void build_send_index()
    {
        const auto duplicates_handler = [this](size_t i, size_t original_i) {
            auto json_body = json_value();
            json_body[JSON_FIELD_DUPLICATE] = original_i;
            answer[i][JSON_FIELD_CODE] = 409;
            answer[i][JSON_FIELD_BODY] = json_body;
        };
        build_send_index(duplicates_handler);
    }

    template <typename F>
    void build_send_index(const F& duplicate_handler)
    {
        // Sort and filter duplicates by uid.
        std::vector<size_t> uid_index = sort_keys_by_uid();
        send_index.reserve(keys.size());
        for (auto it = uid_index.begin(); it != uid_index.end(); ++it)
        {
            if (send_index.empty() || keys[*it].uid != keys[send_index.back()].uid)
            {
                send_index.push_back(*it);
            }
            else
            {
                duplicate_handler(*it, send_index.back());
            }
        }

        // Use only deduplicated array to build final index.
        std::sort(send_index.begin(), send_index.end(), [this](size_t i, size_t j) -> bool {
            return gid_from_uid(keys[i].uid) < gid_from_uid(keys[j].uid);
        });
    }
};

}}}
