#include "batch_send.h"

#include "batch_context.h"
#include "send_parsers.h"
#include "web/methods/hub_routines.h"
#include <yxiva/core/batch_key.h>
#include <yxiva/core/message.h>
#include <yxiva/core/notify_status.h>
#include <yplatform/find.h>

namespace yxiva { namespace web { namespace api2 {

namespace {
bool matchers_supported(const std::vector<filter::subscription_condition>& matchers)
{
    using condition = filter::subscription_condition_type;
    static const std::set<condition> unsupported_matchers = {
        condition::subscription_id, condition::session, condition::uuid, condition::device
    };
    for (auto& value : matchers)
    {
        if (unsupported_matchers.count(value.type))
        {
            return false;
        }
    }
    return true;
}
}

void send_in_parallel(std::shared_ptr<batch_context>& batch_ctx, const settings_ptr& settings);

void send_chunk(
    std::shared_ptr<batch_context>& batch_ctx,
    const settings_ptr& settings,
    size_t chunk_start,
    size_t chunk_end);

void batch_send(
    http_stream_ptr stream,
    settings_ptr settings,
    const service_authorization& auth,
    const string& /*service*/,
    const string& event,
    uint32_t ttl,
    const string& external_request_id)
{
    // TODO: CHECK HTTPS

    auto req = stream->request();

    auto [error, message, attributes] = preparse_message(req);
    if (error) return send_bad_request(stream, error);
    if (message.tags.size() >= settings->api.message_max_tags)
    {
        send_bad_request(stream, "too many tags");
        return;
    }

    if (ttl > settings->api.hub.max_message_ttl)
    {
        return send_bad_request(stream, "invalid ttl");
    }
    ttl = effective_ttl(ttl);

    message.service = auth.service.name;
    message.operation = event;
    message.ttl = ttl;
    if (external_request_id.size())
    {
        message.transit_id += ".";
        message.transit_id += external_request_id;
    }

    batch_keys recipients;
    if (auto error = parse_batch_recipients(attributes, recipients))
    {
        return send_bad_request(stream, error);
    }

    if (recipients.size() > settings->api.hub.batch_max_recipients)
    {
        return send_bad_request(stream, "too many recipients");
    }

    if (!matchers_supported(message.subscription_matchers))
    {
        return send_bad_request(stream, "unsupported subscription matchers for batch");
    }

    message.batch_size = recipients.size();
    stream->ctx()->custom_log_data["batch_size"] = std::to_string(message.batch_size);

    auto batch_ctx =
        std::make_shared<batch_context>(stream, std::move(message), std::move(recipients));

    send_in_parallel(batch_ctx, settings);
}

void send_in_parallel(std::shared_ptr<batch_context>& batch_ctx, const settings_ptr& settings)
{
    size_t total_size = batch_ctx->send_index.size();
    size_t chunk_size = settings->api.hub.batch_size;
    size_t min_chunk_size = settings->api.hub.batch_min_size;

    for (size_t offset = 0U; offset < total_size;)
    {
        size_t next = std::min(offset + chunk_size, total_size);
        if (total_size - next < min_chunk_size)
        {
            next = total_size;
        }
        send_chunk(batch_ctx, settings, offset, next);
        offset = next;
    }
}

void send_chunk(
    std::shared_ptr<batch_context>& batch_ctx,
    const settings_ptr& settings,
    size_t chunk_start,
    size_t chunk_end)
{
    auto context = batch_ctx->stream->request()->context;

    // Prepare keys.
    std::vector<std::reference_wrapper<batch_key>> chunk;
    chunk.reserve(chunk_end - chunk_start);
    for (auto i = chunk_start; i < chunk_end; ++i)
    {
        chunk.push_back(batch_ctx->keys[batch_ctx->send_index[i]]);
    }

    // XXX not optimal
    string body = yxiva::pack(batch_ctx->message);
    body += yxiva::pack(chunk);

    hubrpc::http_options options;
    options.timeouts.total = settings->api.hub.batch_send_timeout;

    auto size = chunk.size();
    string sharding_key = size == 1 ? chunk.front().get().uid : string{};

    string path = "/batch_binary_notify";
    // batch size buckets
    if (size <= settings->api.hub.batch_buckets.small)
    {
        if (size == 1)
        {
            path += "_single/";
        }
        else
        {
            path += "_small/";
        }
    }
    else
    {
        if (size <= settings->api.hub.batch_buckets.medium)
        {
            path += "_medium/";
        }
        else
        {
            path += "/";
        }
    }
    path += yplatform::url_encode(batch_ctx->message.service);

    find_hubrpc()->async_post_binary(
        context,
        sharding_key,
        path,
        {
            { "service", batch_ctx->message.service },
            { "operation", batch_ctx->message.operation },
            { "size", size },
            { "sharding_key", sharding_key } // for debug purpose
        },
        std::move(body),
        options,
        [batch_ctx, chunk_start, chunk_end](
            const boost::system::error_code& err, yhttp::response response) {
            auto context = batch_ctx->stream->request()->context;
            if (err)
            {
                YLOG_CTX_GLOBAL(context, error) << "hub error: " << err.message();
                batch_ctx->report_error(
                    chunk_start, chunk_end, notify_status(500, "internal error"));
            }
            else if (response.status != http_codes::ok)
            {
                YLOG_CTX_GLOBAL(context, notice)
                    << "hub status: " << response.status << " reason: " << response.body;
                if (response.status == http_codes::service_unavailable)
                {
                    batch_ctx->report_error(
                        chunk_start, chunk_end, notify_status(429, "rate limit"));
                }
                else
                {
                    batch_ctx->report_error(
                        chunk_start, chunk_end, notify_status(500, "internal error"));
                }
            }
            else
            {
                std::vector<notify_status> results;
                try
                {
                    unpack(response.body, results);
                }
                catch (const std::exception& e)
                {
                    YLOG_CTX_GLOBAL(context, error) << "hub result unpack exception: " << e.what();
                    batch_ctx->report_error(
                        chunk_start, chunk_end, notify_status(500, "internal error"));
                    return;
                }
                batch_ctx->report(chunk_start, chunk_end, results);
            }
        });
}

}}}
