#include "list.h"

#include "web/auth/api.h"
#include <yxiva/core/callbacks.h>

namespace yxiva { namespace web { namespace api2 {

namespace {

void handle_list_url_proxy_call(
    const boost::system::error_code& err,
    yhttp::response response,
    http_stream_ptr stream)
{
    if (err)
    {
        YLOG_CTX_GLOBAL(stream->ctx(), info) << "hub list failed: error=" << err.message();
        stream->result(http_codes::internal_server_error);
        return;
    }
    if (response.status != 200)
    {
        YLOG_CTX_GLOBAL(stream->ctx(), info)
            << "hub list failed: code=" << response.status << " error=" << response.body;
        stream->result(http_codes::internal_server_error);
        return;
    }

    json_value list;
    if (auto error = list.parse(response.body, json_type::tarray))
    {
        YLOG_CTX_GLOBAL(stream->ctx(), error) << "hub list result parse error: " << *error;
        stream->result(http_codes::internal_server_error);
        return;
    }

    json_value result(json_type::tarray);
    for (auto&& list_item : list.array_items())
    {
        json_value result_item;
        result_item["id"] = list_item["id"];
        result_item["client"] = list_item["client"];
        result_item["session"] = list_item["session_key"];
        auto url = json_get(list_item, "url", string());
        if (callback_uri::is_mobile_uri(url))
        {
            string app;
            string push_token;
            if (callback_uri::parse_mobile_uri(url, app, push_token))
            {
                result_item["app"] = app;
                result_item["uuid"] = list_item["session_key"];
            }
            result_item["platform"] = list_item["platform"];
            result_item["device"] = list_item["device"];
        }
        else if (callback_uri::is_apns_queue_uri(url))
        {
            string service;
            string app;
            string device;
            if (callback_uri::parse_apns_queue_uri(url, service, app, device))
            {
                result_item["app"] = app;
                result_item["device"] = device;
            }
            result_item["platform"] = list_item["platform"];
            result_item["uuid"] = list_item["session_key"];
        }
        else if (!callback_uri::is_webpush_uri(url))
        {
            result_item["url"] = list_item["url"];
        }
        result_item["filter"] = list_item["filter"];
        result_item["ttl"] = list_item["ttl"];
        result_item["extra"] = list_item["extra_data"];
        result_item["init_time"] = list_item["init_time"];
        result_item["last_sent"] = list_item["ack_time"];
        result_item["pos"] = list_item["ack_local_id"];
        result.push_back(result_item);
    }
    stream->set_code(http_codes::ok);
    stream->set_content_type("application/json");
    stream->result_body(json_write_styled(result));
}

}

void list(
    http_stream_ptr stream,
    settings_ptr /*settings*/,
    const service_authorization& auth,
    const string& service_from_req,
    const string& uid,
    const string& topic)
{
    if (uid.empty() && topic.empty())
    {
        send_bad_request(stream, "request should contain \"uid\" or \"topic\" argument");
        return;
    }
    if (uid.size() && topic.size())
    {
        send_bad_request(stream, "request can't contain both \"uid\" and \"topic\" arguments");
        return;
    }

    string service = service_from_req.empty() ? auth.service.name : service_from_req;
    if (auto result = hacks::is_correct_uid(uid, service); !result)
    {
        send_bad_request(stream, "uid is invalid");
        return;
    }

    namespace p = std::placeholders;
    auto metauid = topic.size() ? encode_topic_name(topic) : uid;
    find_hubrpc()->async_get(
        stream->ctx(),
        metauid,
        "/list_json",
        { { "uid", metauid }, { "service", service }, { "priority", "high" } },
        std::bind(handle_list_url_proxy_call, p::_1, p::_2, stream));
}

}}}
