#include "secret_sign.h"

#include "web/auth/secure_sign.h"

#define MIN_SIGN_TTL 2

namespace yxiva { namespace web { namespace api2 {

namespace {

time_t eval_desired_ts(settings_ptr settings, time_t desired_ts)
{
    auto now = std::time(nullptr);
    auto min_ts = now + MIN_SIGN_TTL;
    auto max_ts = now + settings->api.sign_ttl * 3600;
    auto ts = desired_ts ? std::min(std::max(desired_ts, min_ts), max_ts) : max_ts;
    return ts;
}

std::optional<string> check_sign_args(
    const std::vector<string>& services,
    const std::vector<string>& uids,
    const std::vector<string>& topics)
{
    if (topics.size() && uids.size() > 1)
    {
        return "request with topic should contain only one user id";
    }

    if (auto result = hacks::is_correct_uids(uids, services); !result)
    {
        return result.error_reason;
    }

    if (topics.size() && services.size() > 1)
    {
        return "request with topic should contain only one service";
    }

    return std::nullopt;
}

}

void sign(
    http_stream_ptr stream,
    settings_ptr settings,
    const multi_service_authorization& /*auth*/,
    const std::vector<string>& services,
    const std::vector<string>& uids,
    const std::vector<string>& topics,
    std::time_t ts)
{
    if (auto error = check_sign_args(services, uids, topics))
    {
        send_bad_request(stream, *error);
        return;
    }

    ts = eval_desired_ts(settings, ts);
    if (std::time(nullptr) >= ts)
    {
        YLOG_CTX_GLOBAL(stream->ctx(), error) << "invalid expired value";
        stream->result(http_codes::internal_server_error, "failed to create a secret sign");
        return;
    }

    auto sign = make_secure_sign(
        services_uids_topics_data(services, uids, topics), ts, settings->sign_secret);
    stream->set_code(http_codes::ok);
    stream->set_content_type("application/json");
    stream->result_body("{\"sign\": \"" + sign + "\", \"ts\": \"" + std::to_string(ts) + "\"}");
}

void verify_sign(
    http_stream_ptr stream,
    settings_ptr settings,
    const multi_service_authorization& /*auth*/,
    const std::vector<string>& services,
    const std::vector<string>& uids,
    const std::vector<string>& topics,
    const string& sign,
    const std::time_t ts)
{
    if (auto error = check_sign_args(services, uids, topics))
    {
        send_bad_request(stream, *error);
        return;
    }

    auto valid_sign = make_secure_sign(
        services_uids_topics_data(services, uids, topics), ts, settings->sign_secret);
    if (std::time(nullptr) >= ts || valid_sign != sign)
    {
        send_unauthorized(stream, "bad sign");
        return;
    }

    stream->result(http_codes::ok, "");
}

}}}
