#include "send.h"

#include "send_parsers.h"
#include "web/methods/hub_routines.h"
#include <yxiva/core/notify_status_to_json.h>
#include <yxiva/core/message.h>
#include <yxiva/core/notify_status.h>
#include <yxiva/core/operation_result.h>
#include <yplatform/find.h>
#include <random>

namespace yxiva { namespace web { namespace api2 {

namespace {
static const string DM_HEADER = "x-deliverymode";
static const string DM_QUEUE = "queued";
static const string DM_DIRECT = "direct";

std::pair<operation::error, string> extract_delivery_mode(
    const ymod_webserver::request& req,
    const service_properties& service)
{
    auto it = req.headers.find(DM_HEADER);
    if (it != req.headers.end())
    {
        if (it->second != DM_QUEUE && it->second != DM_DIRECT)
        {
            return { "invalid delivery mode \"" + it->second + "\"", ""s };
        }
        return { {}, it->second };
    }
    return { {}, service.queued_delivery_by_default ? DM_QUEUE : DM_DIRECT };
}

void add_headers_from(const http_stream_ptr& stream, const yhttp::response& response)
{
    // ymod_httpclient returns headers in lower case - first element is for lookup.
    // We must respond with capitalized headers - second is for response.
    static const std::vector<std::pair<string, string>> headers_to_pass = {
        { "notificationid", "NotificationID" }, { "transitid", "TransitID" }
    };

    for (auto& header : headers_to_pass)
    {
        auto header_it = response.headers.find(header.first);
        if (header_it != response.headers.end())
        {
            stream->add_header(header.second, header_it->second);
        }
    }
}

json_value raw_report_to_json(const task_context_ptr& ctx, const string& data)
{
    try
    {
        std::vector<notify_status> results;
        unpack(data, results);
        json_value json;
        if (results.size() != 1)
        {
            for (auto& status : results)
            {
                json.push_back(to_json(status));
            }
        }
        else
        {
            json = to_json(results.front());
        }
        return json;
    }
    catch (const std::exception& e)
    {
        YLOG_CTX_GLOBAL(ctx, error) << "report decode exception: " << e.what();
        return to_json(notify_status(500, "internal error"));
    }
}

string generate_stream_uid(uint32_t stream_count)
{
    // Generation is fast, but not thread-safe.
    static thread_local std::minstd_rand engine(std::time(nullptr));
    std::uniform_int_distribution<uint32_t> range(0, stream_count - 1);
    string uid;
    uid.reserve(16);
    uid.append("stream_").append(std::to_string(range(engine)));
    return uid;
}

string effective_delivery_mode(const string& delivery_mode, uint32_t ttl)
{
    if (delivery_mode == DM_QUEUE && 0 == effective_ttl(ttl))
    {
        return DM_DIRECT;
    }
    return delivery_mode;
}

}

void fill_message(
    message& message,
    const string& service,
    const string& uid,
    const string& event,
    uint32_t ttl,
    const string& session,
    const string& external_request_id)
{
    message.ttl = effective_ttl(ttl);
    message.uid = uid;
    message.service = service;
    message.operation = event;
    message.session_key = session;
    if (external_request_id.size())
    {
        message.transit_id += ".";
        message.transit_id += external_request_id;
    }
}

operation::error check_limits(settings_ptr settings, message& message)
{
    if (message.ttl > settings->api.hub.max_message_ttl)
    {
        return "invalid ttl";
    }
    if (message.tags.size() >= settings->api.message_max_tags)
    {
        return "too many tags";
    }
    if (message.raw_data.size() >= settings->api.message_max_payload &&
        !settings->service_features.message_unlimited_payload.enabled_for(message.service))
    {
        return "payload too large";
    }
    return {};
}

std::pair<operation::error, message> create_message(
    ymod_webserver::request_ptr request,
    settings_ptr settings,
    const string& service,
    const string& uid,
    const string& event,
    uint32_t ttl,
    const string& session,
    const string& external_request_id)
{
    auto [error, message, attributes] = preparse_message(request);
    if (error) return { error, {} };

    fill_message(message, service, uid, event, ttl, session, external_request_id);

    if (auto error = check_limits(settings, message))
    {
        return { error, {} };
    }
    return { {}, std::move(message) };
}

void do_send(http_stream_ptr stream, const string& delivery_mode, const message& message)
{
    bool queue_mode = effective_delivery_mode(delivery_mode, message.ttl) == DM_QUEUE;
    bool need_report = delivery_mode == DM_DIRECT /* TODO fix */;

    string sharding_key = message.uid.empty() ? message.service : message.uid;
    string uidservice = message.uid + message.service;
    string path = queue_mode ? "/binary_notify/" : "/fast_binary_notify/";
    path += yplatform::url_encode(message.service);
    find_hubrpc()->async_post_binary(
        stream->ctx(),
        sharding_key,
        path,
        { { "uid", message.uid },
          { "operation", message.operation },
          { "service", message.service },
          { "uidservice", uidservice },
          { "ttl", message.ttl }, // is passed for logging only
          { "report", need_report ? "1" : "0" },
          { "priority", "high" } },
        pack(message),
        [stream, need_report](const boost::system::error_code& err, yhttp::response response) {
            auto req = stream->request();
            if (err || response.status != http_codes::ok)
            {
                handle_default_hub_codes(stream, err, std::move(response));
                return;
            }
            stream->set_code(http_codes::ok);
            add_headers_from(stream, response);
            if (need_report)
            {
                stream->set_content_type("application/json");
                stream->result_body(raw_report_to_json(stream->ctx(), response.body).stringify());
            }
            else
            {
                stream->result_body("");
            }
        });
}

void single_send(
    http_stream_ptr stream,
    settings_ptr settings,
    const service_authorization& auth,
    const string& /*service*/,
    const string& uid,
    const string& topic,
    const string& event,
    uint32_t ttl,
    const string& session,
    const string& external_request_id)
{
    auto req = stream->request();

    // TODO: CHECK HTTPS

    auto [dm_error, delivery_mode] = extract_delivery_mode(*req, auth.service);
    if (dm_error) return send_bad_request(stream, dm_error);

    if (uid.size() && topic.size())
    {
        return send_bad_request(
            stream, "request can't contain both \"uid\" and \"topic\" arguments");
    }

    if (uid.empty() && topic.empty())
    {
        return send_bad_request(
            stream, "request should contain any of \"uid\" or \"topic\" argument");
    }

    // TODO extract
    if (auto result = hacks::is_correct_uid(uid, auth.service.name); !result)
    {
        return send_bad_request(stream, result.error_reason);
    }

    auto [error, message] = create_message(
        req,
        settings,
        auth.service.name,
        topic.size() ? encode_topic_name(topic) : uid,
        event,
        ttl,
        session,
        external_request_id);
    if (error) return send_bad_request(stream, error);

    do_send(stream, delivery_mode, message);
}

void stream_send(
    http_stream_ptr stream,
    settings_ptr settings,
    const service_authorization& auth,
    const string& /*service*/,
    const string& event,
    uint32_t ttl)
{
    auto req = stream->request();

    // TODO: CHECK HTTPS

    auto [dm_error, delivery_mode] = extract_delivery_mode(*req, auth.service);
    if (dm_error) return send_bad_request(stream, dm_error);
    if (effective_delivery_mode(delivery_mode, ttl) == DM_DIRECT)
    {
        send_bad_request(stream, "direct delivery mode is not allowed for stream services");
        return;
    }

    auto uid = generate_stream_uid(auth.service.stream_count);

    auto [error, message] =
        create_message(req, settings, auth.service.name, uid, event, ttl, ""s, ""s);
    if (error) return send_bad_request(stream, error);

    do_send(stream, delivery_mode, message);
}

void wild_send(
    http_stream_ptr stream,
    settings_ptr settings,
    const service_authorization& auth,
    const string& /*service*/,
    const string& event,
    uint32_t ttl,
    const string& session)
{
    auto req = stream->request();
    // TODO: CHECK HTTPS

    auto [dm_error, delivery_mode] = extract_delivery_mode(*req, auth.service);
    if (dm_error) return send_bad_request(stream, dm_error);

    auto [error, message] =
        create_message(req, settings, auth.service.name, ""s, event, ttl, session, ""s);
    if (error) return send_bad_request(stream, error);

    message.broadcast = true;

    do_send(stream, delivery_mode, message);
}

}}}
