#include "common.h"
#include "../arguments/uid.h"
#include <yxiva/core/batch_key.h>
#include <yxiva/core/message.h>
#include <yxiva/core/methods/check.h>
#include <yxiva/core/repacker.h>
#include <yxiva/core/filter/parse.h>
#include <yxiva/core/operation_result.h>
#include <boost/regex.hpp>

namespace yxiva::web::api2 {

inline void extract_matcher(
    ymod_webserver::request_ptr req,
    std::vector<filter::subscription_condition>& matchers)
{
    using cond_t = filter::subscription_condition_type;
    auto& url = req->url;

    string val;
    if ((val = url.param_value("subscription_id")).size())
    {
        matchers.push_back({ cond_t::subscription_id, "", { val } });
    }
    else if ((val = url.param_value("subscription_session")).size())
    {
        matchers.push_back({ cond_t::session, "", { val } });
    }
    else if ((val = url.param_value("subscription_uuid")).size())
    {
        matchers.push_back({ cond_t::uuid, "", { val } });
    }
}

inline bool batch_matchers_supported(const std::vector<filter::subscription_condition>& matchers)
{
    using condition = filter::subscription_condition_type;
    static const std::set<condition> unsupported_matchers = {
        condition::subscription_id, condition::session, condition::uuid, condition::device
    };
    for (auto& value : matchers)
    {
        if (unsupported_matchers.count(value.type))
        {
            return false;
        }
    }
    return true;
}

inline bool validate_tag(const json_value& json_tag)
{
    return validate_param(json_tag.to_string(), tag_char_ex);
}

inline operation::error parse_tags(const json_value& source, std::vector<string>& tags)
{
    auto js_tags = source["tags"];
    if (!js_tags.is_null())
    {
        if (!js_tags.is_array())
        {
            return "invalid tags argument format";
        }
        // ignore empty arrays
        if (js_tags.size())
        {
            tags.reserve(js_tags.size());
            for (auto&& tag : js_tags.array_items())
            {
                if (!tag.is_string() || !validate_tag(tag)) return "invalid tag";
                tags.push_back(tag.to_string());
            }
        }
    }
    return {};
}

inline operation::error parse_keys(const json_value& source, std::map<string, string>& keys)
{
    auto js_keys = source["keys"];
    if (!js_keys.is_null())
    {
        if (!js_keys.is_object())
        {
            return "invalid keys argument format";
        }

        std::vector<string> vector_for_array_value;
        for (auto it = js_keys.members_begin(); it != js_keys.members_end(); ++it)
        {
            auto&& key = it.key();
            auto&& value = (*it);
            if (value.is_string())
            {
                auto value_as_string = value.to_string();
                if (value_as_string != "") keys[string(key)] = value.to_string();
            }
            else if (value.is_array() && value.size())
            {
                vector_for_array_value.reserve(value.size());
                for (auto&& item : value.array_items())
                {
                    if (!item.is_string()) return "not string value " + string(key);
                    vector_for_array_value.push_back(item.to_string());
                }
                std::sort(vector_for_array_value.begin(), vector_for_array_value.end());
                keys[string(key)] = pack(vector_for_array_value);
                vector_for_array_value.clear();
            }
            else
            {
                return "invalid value in keys argument";
            }
        }
    }
    return {};
}

inline operation::error parse_payload(const json_value& source, string& payload)
{
    if (is_null(source))
    {
        return "missing payload";
    }
    if (!source.is_string() && !source.is_object())
    {
        return "unsupported payload format";
    }
    if (source.is_string())
    {
        payload = source.to_string();
    }
    else
    {
        payload = source.stringify();
    }
    return {};
}

inline operation::error parse_repackers(
    const json_value& source,
    std::map<string, string>& repackers,
    const json_value& payload)
{
    static const std::size_t MAX_REPACKER_RAW_SIZE = 4096; // 4KB

    if (!source.has_member("repack")) return {};

    auto&& repackers_js = source["repack"];
    if (!repackers_js.is_object()) return "invalid 'repack' type";

    std::set<string> platforms;
    for (auto it = repackers_js.members_begin(); it != repackers_js.members_end(); ++it)
    {
        auto platform = string(it.key());
        auto&& resolved = platform::resolve_alias(platform);
        if (!resolved.supported && platform != "other")
        {
            return "unknown push services in 'repack'";
        }
        else if (!platforms.insert(resolved.name).second)
        {
            return "duplicate push services in 'repack'";
        }
    }

    for (auto irepacker_js = repackers_js.members_begin();
         irepacker_js != repackers_js.members_end();
         ++irepacker_js)
    {
        auto push_service = string(irepacker_js.key());

        auto&& repacker_js = *irepacker_js;
        custom_repacker repack;
        auto parsed = custom_repacker::from_json(repacker_js, repack);
        if (!parsed)
            return "invalid repacking rules for " + push_service + ": " + parsed.error_reason;

        if (repacker_js.empty()) continue;

        auto validated = repack.validate(payload);
        if (!validated)
            return "invalid repacking rules for " + push_service + ": " + validated.error_reason;

        string packed_repacker = repacker_js.stringify();
        if (packed_repacker.size() > MAX_REPACKER_RAW_SIZE)
            return "invalid repacking rules for " + push_service + ": too large";

        repackers[push_service] = std::move(packed_repacker);
    }
    return {};
}

inline operation::error parse_experiments(const json_value& source, std::set<string>& experiments)
{
    if (!source.has_member("experiments")) return {};

    auto& experiments_js = source["experiments"];
    if (!experiments_js.is_array()) return "invalid json type of \"experiments\"";

    for (auto it = experiments_js.array_begin(); it != experiments_js.array_end(); ++it)
    {
        auto&& item = *it;
        if (!item.is_string()) return "invalid json type of \"experiments\" item";

        static const boost::regex expr("[a-zA-Z0-9]+-[0-9]+");
        if (!boost::regex_match(item.to_string(), expr)) return "invalid experiment id";
        experiments.insert(item.to_string());
    }
    return {};
}

inline operation::error parse_subscription_matchers(
    const json_value& json_matchers,
    std::vector<filter::subscription_condition>& matchers)
{
    if (json_matchers.is_null())
    {
        return {};
    }
    if (!json_matchers.is_array())
    {
        return "invalid json type of \"subscriptions\"";
    }
    for (auto&& json_cond : json_matchers.array_items())
    {
        filter::subscription_condition cond;
        auto condition_decoded = yxiva::filter::decode_condition(cond, json_cond);
        if (!condition_decoded) return condition_decoded.error_reason;

        matchers.emplace_back(std::move(cond));
    }

    return {};
}

inline string to_string(const yplatform::zerocopy::segment& segm)
{
    return string(segm.begin(), segm.end());
}

inline bool is_multipart(const request_ptr& req)
{
    return req->childs.size();
}

inline bool validate_multipart(const request_ptr& req)
{
    // Must contain two parts% json and binary.
    return is_multipart(req) && req->childs.size() == 2 &&
        req->content.mime_type_equals("multipart", "related") &&
        req->childs[0].content.mime_type_equals("application", "json") &&
        req->childs[1].content.mime_type_equals("application", "octet-stream");
}

// TODO extend
inline bool validate_recipient(const batch_key& key)
{
    static const uid_validator validator;
    return key.uid.size() && validator(key.uid);
}

inline operation::error parse_batch_recipients(const json_value& source, batch_keys& result)
{
    auto js_recipients = source["recipients"];
    if (!js_recipients.is_array() || js_recipients.empty()) return "no recipients";

    result.clear();
    result.reserve(js_recipients.size());

    for (auto&& item : js_recipients.array_items())
    {
        batch_key key;
        if (item.is_string())
        {
            key.uid = item.to_string();
        }
        else if (item.is_object() && item.size() == 1 && (*item.members_begin()).is_string())
        {
            auto it = item.members_begin();
            key.uid = string(it.key());
            key.subscription_id = (*it).to_string();
        }
        else
        {
            return "invalid recipient type";
        }

        if (!validate_recipient(key)) return "invalid recipient \"" + key.uid + "\"";

        result.push_back(key);
    }

    return {};
}

inline std::tuple<operation::error, message, json_value> preparse_message(const request_ptr& req)
{
    if (is_multipart(req) && !validate_multipart(req))
    {
        return { "invalid multipart send request", {}, {} };
    }

    json_value attributes;
    yxiva::message message;
    message.event_ts = std::time(nullptr);
    message.event_ts_ms = time_ms();
    message.transit_id = req->context->uniq_id();

    // Locate attributes and body.
    // XXX redundant copy of request body
    string json_src = to_string(is_multipart(req) ? req->childs[0].body : req->raw_body);
    if (auto parsed = json_parse(attributes, json_src); !parsed)
        return { parsed.error_reason, {}, {} };

    if (is_multipart(req))
    {
        message.raw_data = string(req->childs[1].body.begin(), req->childs[1].body.end());
        message.type = message_content_type::binary;
    }
    else
    {
        if (auto error = parse_payload(attributes["payload"], message.raw_data))
            return { error, {}, {} };
        message.type = message_content_type::json;
    }

    if (auto error = parse_tags(attributes, message.tags)) return { error, {}, {} };
    if (auto error = parse_keys(attributes, message.data)) return { error, {}, {} };
    if (message.type != message_content_type::binary)
    {
        if (auto error =
                parse_repackers(attributes, message.repacking_rules, attributes["payload"]))
            return { error, {}, {} };
    }
    if (auto error = parse_experiments(attributes, message.experiments)) return { error, {}, {} };
    if (auto error =
            parse_subscription_matchers(attributes["subscriptions"], message.subscription_matchers))
        return { error, {}, {} };

    if (message.subscription_matchers.empty()) extract_matcher(req, message.subscription_matchers);

    return { {}, message, attributes };
}

}