#include "subscribe_app.h"

#include "web/methods/hub_routines.h"
#include "web/utils/bb_connection_id.h"
#include <yxiva/core/filter.h>
#include <yxiva/core/callbacks.h>
#include <yxiva/core/platforms.h>
#include <yxiva/core/subscription_id.h>
#include <yplatform/yield.h>

namespace yxiva { namespace web { namespace api2 {

using http_response::send_no_argument_error;
namespace p = std::placeholders;

void subscribe_app::operator()(yield_ctx yield_ctx, const error_code& err, yhttp::response response)
{
    if (stream->ctx()->is_cancelled()) return;

    auto user_auth = get_auth_optional<user_authorization>(auth);
    reenter(yield_ctx)
    {
        if (auto result = hacks::is_correct_uid(uid, service.name); !result)
        {
            return send_bad_request(stream, result.error_reason);
        }

        if (user_auth && topic.size())
        {
            return send_bad_request(
                stream, "topics are not supported for OAuth authentication method");
        }

        if (user_auth) uid = user_auth->uid;
        if (uid.empty()) return send_no_argument_error(stream, "user (uid)");
        client = client.empty() ? "mobile" : client;

        if (auto filter_parsed = parse_filter(service.filter, filter, *settings); !filter_parsed)
        {
            return send_bad_request(stream, filter_parsed.error_reason);
        }

        use_apns_queue = platform == platform::APNS_QUEUE;
        // In case queue is disabled, log it and subscribe without queue.
        if (use_apns_queue && !settings->api.apns_queue.enabled)
        {
            use_apns_queue = false;
            YLOG_CTX_GLOBAL(stream->ctx(), warning) << "subscribe_app apns queue is disabled";
        }

        // Queue requires device id.
        if (use_apns_queue && device_id.empty())
        {
            return send_no_argument_error(stream, "device (device_id)");
        }

        // TODO refactor
        {
            auto resolved_platform = ::yxiva::platform::resolve_alias(platform);
            if (!resolved_platform.supported)
            {
                return send_bad_request(stream, "this platform is not supported");
            }
            if (!is_usable_token(push_token, resolved_platform.name))
            {
                return send_bad_request(stream, "invalid pushtoken");
            }

            // Store "real" platform name to find app and subscribe.
            platform = platform::get_real(resolved_platform.name);
        }

        // Only check registration for platforms that require it.
        if (platform != platform::MPNS)
        {
            auto service_manager = find_service_manager(settings->api.auth_service_manager);
            auto app = service_manager->find_app(platform, app_name);
            if (!app || app->secret_key.empty())
            {
                return send_bad_request(
                    stream,
                    "application " + app_name + " for platform " + platform + " is not registered");
            }
        }

        if (use_apns_queue)
        {
            yield do_subscribe_apns_queue(yield_ctx);
            if (err || (response.status && response.status != http_codes::ok))
            {
                return handle_default_hub_codes(stream, err, std::move(response));
            }
            yield do_subscribe_apns_queue_reference(yield_ctx);
        }
        else
        {
            yield do_subscribe_app(yield_ctx);
        }

        respond(err, response);
    }
}

void subscribe_app::do_subscribe_apns_queue(yield_ctx yield_ctx)
{
    // Create impl-subscription: mobile subscription with queue-uid in queue-service,
    // device as uuid, APNS_QUEUE platform and no filters.
    add_subscription(
        { settings->api.apns_queue.service, string{} },
        apns_queue_id(device_id, app_name),
        callback_uri::mobile_uri(app_name, push_token),
        device_id,
        platform::APNS_QUEUE,
        string{},
        "queue",
        device_id,
        string{},
        string{},
        yield_ctx);
}

void subscribe_app::do_subscribe_apns_queue_reference(yield_ctx yield_ctx)
{
    // If impl-subscription was created successfully, then create queue-subscription,
    // which is just as normal mobile subscription created by /subscribe/app,
    // except for a queue-callback.
    add_subscription(
        service,
        uid,
        callback_uri::apns_queue_uri(settings->api.apns_queue.service, app_name, device_id),
        uuid,
        platform::APNS,
        extra,
        client,
        device_id,
        topic,
        bb_connection_id(),
        yield_ctx);
}

void subscribe_app::do_subscribe_app(yield_ctx yield_ctx)
{
    add_subscription(
        service,
        uid,
        callback_uri::mobile_uri(app_name, push_token),
        uuid,
        platform,
        extra,
        client,
        device_id,
        topic,
        bb_connection_id(),
        yield_ctx);
}

void subscribe_app::respond(const error_code& err, const yhttp::response& response)
{
    if (err || response.status != http_codes::ok || !subscription_id_required())
    {
        return handle_default_hub_codes(stream, err, response);
    }

    json_value value;
    value["subscription-id"] = response.body;
    stream->set_code(http_codes::ok);
    stream->set_content_type("application/json");
    stream->result_body(value.stringify());
}

bool subscribe_app::subscription_id_required() const
{
    return settings->service_features.api2_respond_subscription_id.enabled_for(service.name);
}

void subscribe_app::add_subscription(
    const service_with_filter& service,
    const string& uid,
    const string& callback,
    const string& uuid,
    const string& platform,
    const string& extra,
    const string& client,
    const string& device_id,
    const string& topic,
    const string& connection_id,
    yield_ctx yield_ctx)
{
    auto&& metauid = topic.empty() ? uid : encode_topic_name(topic);
    auto&& account = topic.size() ? uid : string();
    string uidservice = metauid + service.name;
    auto subscription_id = make_mobile_subscription_id(uuid);

    find_hubrpc()->async_post(
        stream->ctx(),
        metauid,
        "/subscribe_mobile",
        { { "uid", metauid },
          { "service", service.name },
          { "uidservice", uidservice },
          { "platform", platform },
          { "client", client },
          { "session_key", uuid },
          { "extra_data", extra },
          { "device_id", device_id },
          { "ttl", ttl_for(platform) },
          { "account", account },
          { "bb_connection_id", connection_id },
          { "priority", "high" },
          { "id", subscription_id } },
        yhttp::form_encode({ { "filter", service.filter }, { "callback", callback } }),
        yield_ctx);
}

unsigned subscribe_app::ttl_for(const string& platform)
{
    return platform == ::yxiva::platform::WNS ? settings->api.hub.subscribe_wns_ttl :
                                                settings->api.hub.subscribe_app_ttl;
}

string subscribe_app::bb_connection_id()
{
    auto user_auth = get_auth_optional<user_authorization>(auth);
    return user_auth ? user_auth->bb_connection_id : get_bb_connection_id(stream);
}

}}}
