#include "subscribe_url.h"

#include "web/methods/hub_routines.h"
#include "web/utils/bb_connection_id.h"
#include "web/find_deps.h"
#include <yxiva/core/filter.h>
#include <yxiva/core/subscription_id.h>
#include <yplatform/find.h>

namespace yxiva { namespace web { namespace api2 {

namespace {
void handle_subscribe_url_proxy_call(
    const boost::system::error_code& err,
    yhttp::response response,
    http_stream_ptr stream,
    ttl_t ttl) noexcept
{
    if (err)
    {
        YLOG_CTX_GLOBAL(stream->ctx(), info)
            << "hub subscribe_url failed: exception=" << err.message();
        stream->result(http_codes::internal_server_error);
        return;
    }

    if (response.status == http_codes::ok)
    {
        json_value value;
        value["subscription-id"] = response.body;
        value["ttl"] = ttl;
        stream->set_code(http_codes::ok);
        stream->set_content_type("application/json");
        stream->result_body(value.stringify());
    }
    else
    {
        handle_default_hub_codes(stream, err, response);
    }
}
}

void subscribe_url(
    http_stream_ptr stream,
    settings_ptr settings,
    const service_authorization& auth,
    service_with_filter service,
    const string& callback,
    const string& uid,
    const string& raw_filter,
    const string& raw_client,
    const string& session,
    const string& topic)
{
    if (auto result = hacks::is_correct_uid(uid, service.name); !result)
    {
        send_bad_request(stream, result.error_reason);
        return;
    }

    auto filter_parsed = parse_filter(service.filter, raw_filter, *settings);
    if (!filter_parsed)
    {
        send_bad_request(stream, filter_parsed.error_reason);
        return;
    }

    const string& client = raw_client.size() ? raw_client : auth.client;
    ttl_t ttl = find_processor()->settings()->ttl.get(client);
    auto body = service.filter.empty() ? string() : "filter=" + url_encode(service.filter);
    auto metauid = topic.size() ? encode_topic_name(topic) : uid;
    string uidservice = metauid + service.name;
    auto subscription_id =
        make_subscription_id(metauid, service.name, callback, client, session, service.filter, {});

    namespace p = std::placeholders;
    find_hubrpc()->async_post(
        stream->ctx(),
        metauid,
        "/subscribe",
        { { "uid", metauid },
          { "callback", callback },
          { "service", service.name },
          { "uidservice", uidservice },
          { "client", client },
          { "session_key", session },
          { "ttl", ttl },
          { "account", topic.size() ? uid : string() },
          { "bb_connection_id", get_bb_connection_id(stream) },
          { "id", subscription_id } },
        std::move(body),
        std::bind(handle_subscribe_url_proxy_call, p::_1, p::_2, stream, ttl));
}

}}}
