#include "subscribe_webpush.h"

#include "web/methods/hub_routines.h"
#include "web/utils/bb_connection_id.h"
#include <yxiva/core/filter.h>
#include <yxiva/core/callbacks.h>
#include <yxiva/core/subscription_id.h>
#include <yplatform/yield.h>

namespace yxiva { namespace web { namespace api2 {

void subscribe_webpush::operator()(
    yield_ctx yield_ctx,
    const error_code& err,
    yhttp::response response)
{
    reenter(yield_ctx)
    {
        yield
        {
            if (auto result = hacks::is_correct_uid(uid, service.name); !result)
            {
                send_bad_request(stream, result.error_reason);
                return;
            }

            auto filter_parsed = parse_filter(service.filter, raw_filter, *settings);
            if (!filter_parsed)
            {
                send_bad_request(stream, filter_parsed.error_reason);
                return;
            }

            if (auto result = check_ttl(ttl); !result)
            {
                send_bad_request(stream, result.error_reason);
                return;
            }

            string body;
            body.reserve(service.filter.size() + webpush_subscription.size() + extra.size() + 64);
            body += "filter=";
            body += url_encode(service.filter);
            body += "&callback=";
            body += url_encode(callback_uri::webpush_uri(url_encode(webpush_subscription)));
            body += "&extra_data=";
            body += url_encode(extra);

            auto metauid = topic.empty() ? uid : encode_topic_name(topic);
            string uidservice = metauid + service.name;
            find_hubrpc()->async_post(
                stream->ctx(),
                metauid,
                "/subscribe",
                { { "uid", metauid },
                  { "service", service.name },
                  { "uidservice", uidservice },
                  { "client", client },
                  { "session_key", session },
                  { "ttl", ttl },
                  { "id", make_webpush_subscription_id(session) },
                  { "account", topic.size() ? uid : string() },
                  { "bb_connection_id", get_bb_connection_id(stream) },
                  { "priority", "high" } },
                std::move(body),
                yield_ctx);
        }

        respond(err, response);
    }
}

void subscribe_webpush::respond(const error_code& err, const yhttp::response& response)
{
    if (err || response.status != http_codes::ok || !subscription_id_required())
    {
        return handle_default_hub_codes(stream, err, response);
    }

    json_value value;
    value["subscription-id"] = response.body;
    stream->set_code(http_codes::ok);
    stream->set_content_type("application/json");
    stream->result_body(value.stringify());
}

bool subscribe_webpush::subscription_id_required() const
{
    return settings->service_features.api2_respond_subscription_id.enabled_for(service.name);
}

operation::result subscribe_webpush::check_ttl(uint32_t ttl) const
{
    return ttl > settings->api.hub.subscribe_webpush_ttl ? "ttl is too large" : operation::success;
}

}}}
