#include "watch_subscribers.h"
#include "web/auth/cookie.h"
#include "web/auth/secure_sign.h"
#include <yplatform/yield.h>

namespace yxiva::web::api2 {

void watch_subscribers_coro::operator()(
    yield_ctx yield_ctx,
    const boost::system::error_code& ec,
    yhttp::response response)
{
    reenter(yield_ctx)
    {
        if (uids.empty()) return;

        uids_str.reserve(std::min<size_t>(1024, uids.size() * uids[0].size()));
        for (auto&& uid : uids)
        {
            uids_str += uid;
            uids_str += ",";
        }
        // Remove trailing ','.
        if (uids_str.size())
        {
            uids_str.resize(uids_str.size() - 1);
        }
        batch = uids.size() > 1;
        uids.clear();

        while (true)
        {
            if (*stop) return;
            yield
            {
                subscriber_ptr subscriber = get_subscriber();
                if (!subscriber) return;

                find_hubrpc()->async_get(
                    subscriber->ctx(),
                    {},
                    batch ? "/batch_uids_list_json" : "/list_json",
                    { { "uid", uids_str }, { "service", service }, { "priority", "low" } },
                    yield_ctx);
            }

            if (*stop) return;

            {
                subscriber_ptr subscriber = get_subscriber();
                if (!subscriber) return;

                if (ec || response.status != 200)
                {
                    YLOG_CTX_GLOBAL(subscriber->ctx(), info)
                        << "watch_subscribers list failed: error=" << ec.message()
                        << " status=" << response.status << " body=" << response.body;
                }
                else
                {
                    json_value list;
                    if (auto error = list.parse(response.body, json_type::tarray))
                    {
                        YLOG_CTX_GLOBAL(subscriber->ctx(), error)
                            << "hub list result parse error: " << *error;
                    }
                    else
                    {
                        json_value message;
                        message["event"] = "xiva.subscribers";
                        message["service"] = service;
                        auto&& filtered_list = message["list"];
                        filtered_list.set_array();
                        for (auto&& item : list.array_items())
                        {
                            auto max_age =
                                get_seconds_count(settings->api.watch_subscribers.max_age);
                            if (std::time(nullptr) >= item["init_time"].to_int64() + max_age)
                                continue;
                            json_value result_item;
                            auto&& item_uid = item["uid"].to_string();
                            if (is_topic_name(item_uid))
                            {
                                string topic_name;
                                topic_name += decode_topic_name(item_uid);
                                result_item["topic"] = topic_name;
                            }
                            else
                            {
                                result_item["uid"] = item_uid;
                            }
                            // result_item["alive"] = item["init_time"]; // this breaks last_sent
                            result_item["client"] = item["client"];
                            result_item["session"] = item["session_key"];
                            filtered_list.push_back(result_item);
                        }

                        auto message_str = message.stringify();
                        if (last_sent != message_str)
                        {
                            last_sent = message_str;
                            subscriber->notify_plain_text(message_str);
                        }
                    }
                }
            }

            if (*stop) return;

            yield
            {
                auto timer = std::make_shared<steady_timer>(*io);
                timer->expires_from_now(settings->api.watch_subscribers.list_interval);
                timer->async_wait([timer, yield_ctx](auto ec) { yield_ctx(ec); });
            }
        }
    }
}

void watch_subscribers_impl(
    websocket_stream_ptr stream,
    settings_ptr settings,
    const auth_info& /* ignored */,
    const string& service,
    std::vector<string> topics)
{
    if (topics.empty())
    {
        send_bad_request(stream, "empty topics list");
        return;
    }

    if (topics.size() > settings->api.watch_subscribers.max_topics)
    {
        send_bad_request(stream, "too many topics");
        return;
    }

    if (!settings->api.watch_subscribers.enabled_for.count(service))
    {
        send_forbidden(stream, "");
        return;
    }

    for (auto& topic : topics)
    {
        topic = encode_topic_name(topic);
    }

    // XXX copy-paste
    auto formatters_kit = find_processor()->formatters();
    if (!formatters_kit->has("json"))
    {
        send_internal_error(stream, "formatter was not found");
        return;
    }
    auto formatter = formatters_kit->get("json");

    auto subscriber = boost::make_shared<websocket_subscriber>(
        stream->ctx(), stream, formatter, 0, time_duration::max(), stream->ctx()->uniq_id());

    subscriber->init();

    auto watch_coro = std::make_shared<watch_subscribers_coro>(
        watch_subscribers_coro{ stream->get_io_service(),
                                service,
                                std::move(topics),
                                [subscriber]() { return subscriber; },
                                settings });

    auto future_close = make_future_close(stream);
    future_close.add_callback([future_close, watch_coro]() { *watch_coro->stop = true; });

    yplatform::spawn(stream->get_io_service().get_executor(), watch_coro);
}

watch_subscribers::watch_subscribers(settings_ptr settings) : settings(settings)
{
    continue_with_cookie_auth = auth_with_cookie_websocket(settings, watch_subscribers_impl);
}

void watch_subscribers::operator()(
    websocket_stream_ptr stream,
    const string& service,
    const std::vector<string>& topics,
    const string& uid,
    const string& secret_sign,
    const std::time_t sign_ts) const
{
    if (secret_sign.size())
    {
        auto valid_sign = make_secure_sign(
            services_uids_topics_data({ service }, { uid }, topics),
            sign_ts,
            settings->sign_secret);
        if (std::time(nullptr) >= sign_ts || valid_sign != secret_sign)
        {
            send_unauthorized(stream, "bad sign");
            return;
        }

        auth_info dummy;
        watch_subscribers_impl(stream, settings, dummy, service, topics);
    }
    else
    {
        continue_with_cookie_auth(stream, service, topics);
    }
}

}

#include <yplatform/unyield.h>