#pragma once

#include "web/stream_helpers.h"
#include "web/types.h"
#include "web/util.h"
#include <yplatform/util/safe_call.h>
#include <functional>

namespace yxiva { namespace web {

template <typename Handler, typename... Args>
void handle_abc_response(
    http_stream_ptr stream,
    settings_ptr settings,
    const auth_info& auth,
    const boost::system::error_code& err,
    yhttp::response resp,
    Handler&& handler,
    Args&&... args)
{
    if (err || resp.status != 200)
    {
        YLOG_CTX_GLOBAL(stream->ctx(), error)
            << "abc request failed: \""
            << (err ? err.message() : "status " + std::to_string(resp.status)) << "\"";
        return stream->result(http_codes::unauthorized, "authorization failed");
    }

    try
    {
        json_value resp_json;
        if (auto error = resp_json.parse(resp.body))
        {
            throw std::runtime_error(*error);
        }
        std::vector<string> abc_services;
        abc_services.reserve(resp_json["results"].size());
        for (auto&& result : resp_json["results"].array_items())
        {
            auto service = result["service"]["slug"].to_string();
            if (service.size()) abc_services.push_back(service);
        }
        attach_custom_data(stream, "abc_services", abc_services);
        yplatform::safe_call(
            stream->ctx(),
            "abc authorization handler",
            handler,
            stream,
            settings,
            auth,
            abc_services,
            std::forward<Args>(args)...);
    }
    catch (const std::exception& ex)
    {
        YLOG_CTX_GLOBAL(stream->ctx(), error)
            << "failed to process abc response: \"" << ex.what() << "\"";
        stream->result(http_codes::unauthorized, "authorization failed");
    }
}

template <typename Handler, typename... Args>
void auth_with_abc(
    http_stream_ptr stream,
    settings_ptr settings,
    const auth_info& auth,
    Handler&& handler,
    Args&&... args)
{
    namespace p = std::placeholders;

    if (!settings->api.abc.enabled)
    {
        return yplatform::safe_call(
            stream->ctx(),
            "abc authorization handler",
            handler,
            stream,
            settings,
            auth,
            std::vector<string>{},
            std::forward<Args>(args)...);
    }
    yplatform::find<yhttp::cluster_client>("abc_client")
        ->async_run(
            stream->ctx(),
            yhttp::request::GET(
                "/api/v4/services/members/" +
                    ymod_httpclient::url_encode({ { "person__uid", auth.users.front().uid },
                                                  { "role__code__in", settings->api.abc.roles },
                                                  { "fields", "service.slug" },
                                                  { "page_size", 1000 } }),
                "Authorization: OAuth " + settings->api.abc.oauth_token + "\r\n"),
            [stream, settings, auth, handler = std::forward<Handler>(handler), args...](
                const boost::system::error_code& err, yhttp::response resp) {
                handle_abc_response(stream, settings, auth, err, resp, handler, args...);
            });
}

template <typename Handler>
auto abc_auth(Handler&& handler)
{
    return
        [handler = std::forward<Handler>(handler)](
            http_stream_ptr stream, settings_ptr settings, const auth_info& auth, auto&&... args) {
            auth_with_abc(stream, settings, auth, handler, std::forward<decltype(args)>(args)...);
        };
}

}}
