#pragma once

#include "web/auth/oauth.h"
#include "web/auth/methods/xtoken.h"
#include "web/auth/methods/oauth.h"
#include "web/auth/methods/tvm.h"
#include "web/auth/methods/chain.h"
#include "web/settings.h"
#include "web/stream_helpers.h"
#include "web/util.h"

#include <boost/algorithm/string/predicate.hpp>
#include <algorithm>

namespace yxiva { namespace web {
namespace detail {

template <typename StreamPtr>
inline void protect_token_parameter(const StreamPtr& stream)
{
    using namespace std::string_literals;
    auto& url = stream->request()->url;
    for (auto&& alias : { "token"s, "stoken"s, "ctoken"s })
    {
        auto it = url.params.find(alias);
        if (it != url.params.end())
        {
            protect_request_data(stream, it->second);
        }
    }
}

template <typename StreamPtr>
inline void log_services(const StreamPtr& stream, const std::vector<string>& services)
{
    auto& custom_log_data = stream->request()->context->custom_log_data;
    // Do not overwrite services logged by token auth method.
    if (services.size() && custom_log_data.count("service") == 0)
    {
        custom_log_data["service"] = to_comma_separated_list(services);
    }
}

template <typename StreamPtr>
inline string get_auth_error_reason(const boost::system::error_code& ec, const StreamPtr& stream)
{
    if (ec == make_error(auth_error::multi_token_error))
    {
        auto& custom_log_data = stream->request()->context->custom_log_data;
        auto it = custom_log_data.find("error");
        return it != custom_log_data.end() ? it->second : ec.message();
    }
    // Compatibility: if no credentials could be found,
    // or a generic error occured - provide no custom reason,
    // body of 401 should contain just "Unauthorized".
    return ec == make_error(auth_error::no_credentials) ||
            ec == make_error(auth_error::authorization_error) ?
        string{} :
        ec.message();
}

template <typename Method, typename NextHandler, bool first_arg_is_service>
struct auth
{
    using method = typename std::decay<Method>::type;
    using next_handler = typename std::decay<NextHandler>::type;

    settings_ptr settings;
    method auth_method;
    next_handler next;

    auth(const settings_ptr& st, Method&& auth_method, NextHandler&& next)
        : settings(st)
        , auth_method(std::forward<Method>(auth_method))
        , next(std::forward<NextHandler>(next))
    {
    }

    template <typename StreamPtr, typename... Args>
    void operator()(const StreamPtr& stream, Args&&... args)
    {
        auto bound_next =
            std::bind(next, stream, settings, std::placeholders::_1, std::forward<Args>(args)...);
        auto services = extract_services(std::forward<Args>(args)...);
        auth_method(
            settings,
            stream,
            services,
            [stream, services, bound_next](auto&& ec, auto&& authorization) mutable {
                protect_token_parameter(stream);
                log_services(stream, services);
                if (ec)
                {
                    send_unauthorized(stream, get_auth_error_reason(ec, stream));
                    return;
                }
                bound_next(authorization);
            });
    }

private:
    template <typename... Args>
    std::vector<string> extract_services(Args&&... args)
    {
        if constexpr (first_arg_is_service)
        {
            return extract_service_arg(std::forward<Args>(args)...);
        }
        return {};
    }

    template <typename... Args>
    std::vector<string> extract_service_arg(const string& service, Args&&...)
    {
        return { service };
    }

    template <typename... Args>
    std::vector<string> extract_service_arg(const std::vector<string>& services, Args&&...)
    {
        return services;
    }

    template <typename... Args>
    std::vector<string> extract_service_arg(const service_with_filter& service, Args&&...)
    {
        return { service.name };
    }
};

}

template <typename Authorization, bool first_arg_is_service, typename... Methods>
auto auth_factory(const settings_ptr& settings, Methods&&... methods)
{
    auto factory = [=](auto&& next) mutable {
        auto method = auth::chain<Authorization, Methods...>(std::forward<Methods>(methods)...);
        return detail::auth<decltype(method), decltype(next), first_arg_is_service>(
            settings, std::move(method), std::forward<decltype(next)>(next));
    };
    return factory;
}

template <typename LHS, typename RHS>
struct any_from
{
    std::decay_t<LHS> lhs_method;
    std::decay_t<RHS> rhs_method;

    any_from(const LHS& lhs, const RHS& rhs) : lhs_method(lhs), rhs_method(rhs)
    {
    }

    template <typename StreamPtr, typename Handler>
    void operator()(
        const settings_ptr& settings,
        const StreamPtr& stream,
        const std::vector<string>& service_names,
        Handler&& handler)
    {
        lhs_method(
            settings,
            stream,
            service_names,
            [rhs_method_ = rhs_method,
             handler_ = std::forward<Handler>(handler),
             settings,
             stream,
             service_names](auto&& ec, auto&& auth) mutable {
                if (!ec)
                {
                    handler_(ec, auth);
                    return;
                }
                rhs_method_(settings, stream, service_names, std::move(handler_));
            });
    }
};

}}
