#pragma once

#include "user.h"
#include "web/settings.h"
#include "web/stream_helpers.h"
#include "web/util.h"
#include <boost/range/adaptor/transformed.hpp>

namespace yxiva { namespace web {

template <typename Handler, typename UnauthorizedHandler>
class cookie_auth
{
    using handler_t = typename std::decay<Handler>::type;
    using unauthorized_handler_t = typename std::decay<UnauthorizedHandler>::type;
    using this_type = cookie_auth<Handler, UnauthorizedHandler>;

public:
    cookie_auth(settings_ptr settings, Handler&& h, UnauthorizedHandler unauthorized_handler)
        : settings(std::move(settings))
        , handler(std::forward<Handler>(h))
        , unauthorized_handler(std::move(unauthorized_handler))
    {
        http_client = yplatform::find<yhttp::cluster_client, std::shared_ptr>("akita_client");
    }

    template <typename Stream, typename... Args>
    void operator()(Stream stream, Args&&... args)
    {
        namespace p = std::placeholders;

        auto req = stream->request();
        if (!req->origin || !settings->origin_domains.check_allowed(req->origin->domain).first)
        {
            // User-defined handle_unauthorized is useless in case of wrong origin
            send_unauthorized(stream, "this origin domain is not allowed");
            return;
        }
        auto ctx = req->context;
        if (req->headers["cookie"].empty())
        {
            handle_unauthorized(stream);
            return;
        }
        ctx->profilers.push("passport::cookie");
        // No shared_from_this, because handle_cookie_check is intended
        // to be stored by ymod_webserver by bind() call.
        auth_user_cookies(
            http_client,
            ctx,
            req->headers["cookie"],
            req->headers["host"],
            { ctx->remote_address, ctx->remote_port },
            std::bind(
                &this_type::handle_cookie_check<Stream, Args...>,
                this,
                p::_1,
                p::_2,
                stream,
                std::forward<Args>(args)...));
    }

private:
    template <typename Stream>
    void handle_unauthorized(const Stream& stream)
    {
        try
        {
            unauthorized_handler(stream);
        }
        catch (const std::exception& ex)
        {
            YLOG_CTX_LOCAL(stream->request()->context, error)
                << "unauthorized handler exception: " << ex.what();
        }
        catch (...)
        {
            YLOG_CTX_LOCAL(stream->request()->context, error)
                << "unknown exception unauthorized request";
        }
    }

    template <typename Stream, typename... Args>
    void handle_cookie_check(
        const auth_error::error_code& err,
        const auth_info& auth,
        Stream stream,
        Args&&... args)
    {
        using boost::adaptors::transformed;

        auto ctx = stream->request()->context;
        ctx->profilers.pop("passport::cookie");
        if (err)
        {
            handle_unauthorized(stream);
            return;
        }
        try
        {
            attach_custom_data(stream, "uids", auth.users | transformed([](const user_info& info) {
                                                   return static_cast<const string&>(info.uid);
                                               }));
            handler(stream, settings, auth, std::forward<Args>(args)...);
        }
        catch (const std::exception& ex)
        {
            YLOG_CTX_LOCAL(ctx, error) << "request handler exception: " << ex.what();
        }
        catch (...)
        {
            YLOG_CTX_LOCAL(ctx, error) << "unknown exception handling the request";
        }
    }

    settings_ptr settings;
    handler_t handler;
    unauthorized_handler_t unauthorized_handler;
    std::shared_ptr<yhttp::cluster_client> http_client;
};

template <typename Handler>
auto auth_with_cookie(settings_ptr settings, Handler&& handler)
{
    auto default_noauth_handler = [](http_stream_ptr stream) { send_unauthorized(stream, ""); };
    return cookie_auth(
        std::move(settings), std::forward<Handler>(handler), std::move(default_noauth_handler));
}

template <typename Handler>
auto auth_with_cookie_websocket(settings_ptr settings, Handler&& handler)
{
    auto default_noauth_handler = [](websocket_stream_ptr stream) {
        send_unauthorized(stream, "");
    };
    return cookie_auth(
        std::move(settings), std::forward<Handler>(handler), std::move(default_noauth_handler));
}

template <typename Handler, typename UnauthorizedHandler>
auto auth_with_cookie(
    settings_ptr settings,
    Handler&& handler,
    UnauthorizedHandler&& unauthorized_handler)
{
    return cookie_auth(
        std::move(settings), std::forward<Handler>(handler), std::move(unauthorized_handler));
}
}}
