#pragma once

#include <boost/system/error_code.hpp>
#include <string>
#include <vector>

namespace yxiva { namespace web { namespace auth_error {

using boost::system::error_code;

enum code : int
{
    success = 0,
    no_user_id,
    request_failed,
    bad_host_response,
    authorization_error,
    no_account_info,
    empty_user_info,
    no_credentials,
    bad_token,
    forbidden_service,
    empty_service_list,
    multi_token_error,
    invalid_tvm_ticket,
    unknown_service,
    too_many_services,
    unknown_error,

    COUNT
};

}}}

namespace boost { namespace system {

template <>
struct is_error_code_enum<yxiva::web::auth_error::code>
{
    static constexpr bool value = true;
};

} // namespace system
} // namespace boost

namespace yxiva { namespace web { namespace auth_error {

class error_category : public boost::system::error_category
{
public:
    const char* name() const noexcept
    {
        return "auth";
    }

    std::string message(int ev) const
    {
        static constexpr std::initializer_list<const char*> names = {
            "success",
            "no user id",
            "request to the auth host failed",
            "bad response from the auth host",
            "auth error",
            "no account info",
            "empty user info",
            "no credentials",
            "bad token",
            "forbidden service",
            "empty service list",
            "multi token error",
            "invalid tvm ticket",
            "unknown service",
            "too many services",
            "unknown error",
        };
        static_assert(
            names.size() == code::COUNT,
            "Error codes count doesn't correspond with error names count");
        static const std::vector<std::string> messages(names.begin(), names.end());

        return messages[ev];
    }

    static const error_category& instance()
    {
        static const error_category category;
        return category;
    }

private:
    error_category() = default;
};

inline error_code make_error(code err)
{
    return error_code(err, error_category::instance());
}

}}}
