#pragma once

#include "web/settings.h"
#include "web/util.h"
#include "web/auth/authorization.h"
#include "web/auth/error.h"
#include "service_manager/interface.h"
#include <ymod_tvm/module.h>

namespace yxiva::web::auth {

enum class tvm_check_type
{
    publisher,
    stream_publisher,
    subscriber,
    multi_subscriber
};

template <tvm_check_type check_type, typename TVMModule = ymod_tvm::tvm2_module>
struct tvm
{
    using tvm_module_ptr = std::shared_ptr<TVMModule>;
    using authorization = std::conditional_t<
        check_type == tvm_check_type::multi_subscriber,
        multi_service_authorization,
        service_authorization>;

    tvm_module_ptr tvm_module;
    service_manager_ptr service_manager;

    template <typename StreamPtr, typename Handler>
    void operator()(
        const settings_ptr&,
        const StreamPtr& stream,
        const std::vector<string>& service_names,
        Handler&& handler)
    {
        auto ticket = extract_ticket(stream);
        if (ticket.empty())
        {
            handler(make_error(auth_error::no_credentials), authorization{});
            return;
        }
        auto [check_result, src, issuer_uid] = check_ticket(stream->ctx(), ticket);
        if (!check_result)
        {
            log_and_fail(auth_error::invalid_tvm_ticket, check_result, stream, handler);
            return;
        }
        log_src(src, stream);
        log_issuer(issuer_uid, stream);
        if (service_names.empty())
        {
            handler(make_error(auth_error::empty_service_list), authorization{});
            return;
        }
        auto [find_result, services] = find_services(service_names);
        if (!find_result)
        {
            log_and_fail(auth_error::unknown_service, find_result, stream, handler);
            return;
        }
        log_src_names(resolve(src, services), stream);
        if (auto check_allowed = check_src_allowed(src, services); !check_allowed)
        {
            log_and_fail(auth_error::forbidden_service, check_allowed, stream, handler);
            return;
        }
        auto [make_result, auth] = make<authorization>(services, std::to_string(src));
        if (!make_result)
        {
            log_and_fail(auth_error::too_many_services, make_result, stream, handler);
            return;
        }
        handler(boost::system::error_code{}, std::move(auth));
    }

private:
    using service_data_ptr = std::shared_ptr<const service_data>;
    using service_list = std::vector<service_data_ptr>;
    using string_set = std::unordered_set<string>;
    using app_set = std::set<tvm_app_info>;
    using apps_map = std::map<string, app_set>;

    template <typename StreamPtr>
    string extract_ticket(const StreamPtr& stream)
    {
        auto& headers = stream->request()->headers;
        auto it = headers.find("x-ya-service-ticket");
        return it != headers.end() ? it->second : "";
    }

    std::tuple<operation::result, uint32_t, uint64_t> check_ticket(
        const task_context_ptr& ctx,
        const string& ticket)
    {
        auto native_ticket = tvm_module->get_native_service_ticket(ctx, ticket);
        if (!native_ticket)
        {
            return { "empty ticket", {}, {} };
        }
        if (!*native_ticket)
        {
            ymod_tvm::error_code err = native_ticket->GetStatus();
            return { err.message(), {}, {} };
        }
        return { {}, native_ticket->GetSrc(), native_ticket->GetIssuerUid() };
    }

    std::tuple<operation::result, service_list> find_services(
        const std::vector<string>& service_names)
    {
        service_list services;
        services.reserve(service_names.size());
        for (auto&& service_name : service_names)
        {
            auto service = service_manager->find_service_by_name(service_name);
            if (!service)
            {
                return { "no such service " + service_name, {} };
            }
            services.push_back(service);
        }
        return { {}, services };
    }

    auto resolve(uint32_t app_id, const service_list& services)
    {
        string_set app_names;
        for (auto&& service : services)
        {
            auto [res, app] = find_app(app_id, service->properties);
            if (res && app->name.size())
            {
                app_names.insert(app->name);
            }
        }
        return app_names;
    }

    operation::result check_src_allowed(uint32_t app_id, const service_list& services)
    {
        for (auto&& service : services)
        {
            auto [res, app] = find_app(app_id, service->properties);
            if (!res)
            {
                return res;
            }
            if (app->suspended)
            {
                return "app " + app->name + " suspended";
            }
            if (publisher_auth(check_type))
            {
                if (auto res = check_publish_allowed(service->properties); !res)
                {
                    return res;
                }
            }
        }
        return {};
    }

    auto find_app(uint32_t app_id, const service_properties& service)
    {
        switch (check_type)
        {
        case tvm_check_type::publisher:
        case tvm_check_type::stream_publisher:
            return find_app(app_id, service.name, service.tvm_publishers);
        case tvm_check_type::subscriber:
        case tvm_check_type::multi_subscriber:
            return find_app(app_id, service.name, service.tvm_subscribers);
        }
    }

    operation::result check_publish_allowed(const service_properties& service)
    {
        bool stream_requested = check_type == tvm_check_type::stream_publisher;
        if (!service.is_stream && stream_requested)
        {
            return "service " + service.name + " is not stream";
        }
        if (service.is_stream && !stream_requested)
        {
            return "service " + service.name + " is stream";
        }
        return {};
    }

    std::tuple<operation::result, app_set::iterator> find_app(
        uint32_t app_id,
        const string& service_name,
        const apps_map& apps_by_environment)
    {
        auto&& environment = service_manager->environment();
        auto it = apps_by_environment.find(environment);
        if (it == apps_by_environment.end())
        {
            return { "no tvm apps in environment", {} };
        }
        auto&& apps = it->second;
        auto app = apps.find({ app_id, {} });
        if (app == apps.end())
        {
            auto app_str = std::to_string(app_id);
            return { "tvm_src " + app_str + " not allowed in service " + service_name, {} };
        }
        return { {}, app };
    }

    template <typename Authorization>
    std::tuple<operation::result, Authorization> make(const service_list&, const string&)
    {
        return { "unknown authorization type", {} };
    }

    template <>
    std::tuple<operation::result, service_authorization> make<service_authorization>(
        const service_list& services,
        const string& client)
    {
        if (services.size() > 1)
        {
            return { "too many services", {} };
        }
        return { {}, { services[0]->properties, client } };
    }

    template <>
    std::tuple<operation::result, multi_service_authorization> make<multi_service_authorization>(
        const service_list& services,
        const string& client)
    {
        multi_service_authorization auth;
        auth.reserve(services.size());
        for (auto&& service : services)
        {
            auth.push_back({ service->properties, client });
        }
        return { {}, std::move(auth) };
    }

    template <typename StreamPtr, typename Handler>
    void log_and_fail(
        auth_error::code c,
        const operation::result& result,
        const StreamPtr& stream,
        Handler&& handler)
    {
        auto& reason = result.error_reason;
        auto& ctx = stream->request()->context;
        YLOG_CTX_GLOBAL(ctx, info) << "tvm unauthorized: "
                                      "reason=\""
                                   << reason << "\"";
        ctx->custom_log_data["error"] = reason;
        handler(make_error(c), authorization{});
    }

    template <typename StreamPtr>
    void log_src(uint32_t src, const StreamPtr& stream)
    {
        stream->request()->context->custom_log_data["tvm_src"] = std::to_string(src);
    }

    template <typename StreamPtr>
    void log_src_names(const string_set& src_names, const StreamPtr& stream)
    {
        if (src_names.size())
        {
            stream->request()->context->custom_log_data["tvm_src_name"] =
                to_comma_separated_list(src_names);
        }
    }

    template <typename StreamPtr>
    void log_issuer(const uint32_t& issuer_uid, const StreamPtr& stream)
    {
        if (issuer_uid > 0)
        {
            stream->request()->context->custom_log_data["tvm_uid"] = std::to_string(issuer_uid);
        }
    }

    bool publisher_auth(tvm_check_type c)
    {
        return c == tvm_check_type::publisher || c == tvm_check_type::stream_publisher;
    }
};

}