#pragma once

#include "web/settings.h"
#include "web/util.h"
#include "web/auth/authorization.h"
#include "web/auth/methods/get_token.h"
#include "web/auth/error.h"
#include "service_manager/interface.h"
#include "yxiva/core/split.h"

#include <iostream>

namespace yxiva::web::auth {
namespace detail {
struct check_token_trace
{
    string token_names;
    string service_names;
    void append(const string& token_name, const string& service_name)
    {
        append_to_comma_separated_list(token_names, token_name);
        append_to_comma_separated_list(service_names, service_name);
    }
};

class string_set
{
    std::vector<string> strings;
    bool distinctive = true;

    void make_sorted_and_unique()
    {
        std::sort(strings.begin(), strings.end());
        strings.erase(std::unique(strings.begin(), strings.end()), strings.end());
        distinctive = true;
    }

public:
    void insert(const string& str)
    {
        strings.push_back(str);
        distinctive = false;
    }

    template <typename InputIt>
    void insert(InputIt first, InputIt last)
    {
        strings.insert(strings.end(), first, last);
        distinctive = false;
    }

    template <typename InputIt>
    string missing_amongst(InputIt first, InputIt last)
    {
        if (!distinctive) make_sorted_and_unique();
        string missing_strings;
        for (auto it = first; it != last; ++it)
        {
            const auto& str = *it;
            if (!std::binary_search(strings.begin(), strings.end(), str))
            {
                append_to_comma_separated_list(missing_strings, str);
            }
        }
        return missing_strings;
    }
};
}

enum class xtoken_check_type
{
    send,
    send_stream,
    listen,
    listen_no_service
};

template <xtoken_check_type token_check_type>
struct xtoken
{
    service_manager_ptr service_manager;
    std::vector<string> token_aliases = { "token" };

    template <typename StreamPtr, typename Handler>
    void operator()(
        const settings_ptr& settings,
        const StreamPtr& stream,
        const std::vector<string>& service_names,
        Handler&& handler)
    {
        auto tokens = parse_tokens(stream);
        if (tokens.empty())
        {
            handler(make_error(auth_error::no_credentials), service_authorization{});
            return;
        }
        if (tokens.size() > 1)
        {
            log_unauthorized("multiple tokens not supported", stream);
            handler(make_error(auth_error::bad_token), service_authorization{});
            return;
        }
        if (token_check_type == xtoken_check_type::listen && service_names.empty())
        {
            log_unauthorized("empty service list", stream);
            handler(make_error(auth_error::empty_service_list), service_authorization{});
            return;
        }
        auto [find_result, service, token_properties] = find_token_properties(tokens[0]);
        if (!find_result)
        {
            log_unauthorized(find_result.error_reason, stream);
            handler(make_error(auth_error::bad_token), service_authorization{});
            return;
        }
        auto check_result = settings->api.strict_token_check_mode ?
            check_services(service_names, token_properties) :
            operation::result{};
        if (!check_result)
        {
            log_unauthorized(check_result.error_reason, stream);
            handler(make_error(auth_error::forbidden_service), service_authorization{});
            return;
        }
        log_services_tokens({ get_name(token_properties), service->properties.name }, stream);
        handler(
            boost::system::error_code{},
            service_authorization{ service->properties, get_client(token_properties) });
    }

protected:
    using service_data_ptr = std::shared_ptr<const service_data>;
    using token_properties =
        std::variant<const send_token_properties*, const listen_token_properties*>;
    using service_token_pair = std::pair<service_data_ptr, token_properties>;

    template <typename StreamPtr>
    std::vector<string> parse_tokens(const StreamPtr& stream)
    {
        static const string XIVA_AUTH_PREFIX = "xiva ";

        auto token_str = get_token_from_header(stream, XIVA_AUTH_PREFIX);
        auto token_str_from_parameter = get_token_from_parameter(stream);
        if (token_str.empty())
        {
            token_str = token_str_from_parameter;
        }
        return utils::split_unique(token_str, ",");
    }

    template <typename StreamPtr>
    string get_token_from_parameter(const StreamPtr& stream)
    {
        auto& url = stream->request()->url;
        for (auto&& alias : token_aliases)
        {
            auto it = url.params.find(alias);
            if (it != url.params.end())
            {
                return it->second;
            }
        }
        return {};
    }

    std::tuple<operation::result, service_data_ptr, token_properties> find_token_properties(
        const string& token)
    {
        auto service = find_service_data(token);
        if (!service)
        {
            return { "no service for token", {}, {} };
        }
        auto token_properties = find_token_properties(token, service);
        if (empty(token_properties))
        {
            return { "no properties for token", {}, {} };
        }
        return { operation::result{}, service, token_properties };
    }

    auto find_service_data(const string& token)
    {
        switch (token_check_type)
        {
        case xtoken_check_type::send:
        case xtoken_check_type::send_stream:
            return find_service_by_send_token(token);
        case xtoken_check_type::listen:
        case xtoken_check_type::listen_no_service:
            return service_manager->find_service_by_listen_token(token);
        }
    }

    service_data_ptr find_service_by_send_token(const string& token)
    {
        auto service_data = service_manager->find_service_by_send_token(token);
        bool is_stream = token_check_type == xtoken_check_type::send_stream;
        if (service_data && is_stream != service_data->properties.is_stream)
        {
            return nullptr;
        }
        return service_data;
    }

    token_properties find_token_properties(const string& token, const service_data_ptr& service)
    {
        switch (token_check_type)
        {
        case xtoken_check_type::send:
        case xtoken_check_type::send_stream:
            return find_token_properties(
                token, service_manager->environment(), service->send_tokens);
        case xtoken_check_type::listen:
        case xtoken_check_type::listen_no_service:
            return find_token_properties(
                token, service_manager->environment(), service->listen_tokens);
        }
    }

    template <typename TokenPropertiesMap>
    auto find_token_properties(
        const string& token,
        const string& environment,
        const TokenPropertiesMap& token_properties_map)
    {
        auto it = token_properties_map.find({ token, environment });
        const typename TokenPropertiesMap::mapped_type* ret =
            it != token_properties_map.end() ? &(it->second) : nullptr;
        return ret;
    }

    operation::result check_services(
        const std::vector<string>& service_names,
        const token_properties& token_properties)
    {
        switch (token_check_type)
        {
        case xtoken_check_type::send:
        case xtoken_check_type::send_stream:
        case xtoken_check_type::listen_no_service:
            return {};
        case xtoken_check_type::listen:
            return check_listen_allowed(service_names, token_properties);
        }
    }

    operation::result check_listen_allowed(
        const std::vector<string>& services,
        const token_properties& token_properties)
    {
        auto ltoken_properties = std::get<const listen_token_properties*>(token_properties);
        for (auto&& service : services)
        {
            if (!ltoken_properties->is_listen_allowed(service))
            {
                return "listen not allowed for " + service;
            }
        }
        return {};
    }

    bool empty(const token_properties& props)
    {
        return std::get_if<const listen_token_properties*>(&props) == nullptr &&
            std::get_if<const send_token_properties*>(&props) == nullptr;
    }

    string get_name(const token_properties& props)
    {
        return std::visit([](auto arg) { return arg->name; }, props);
    }

    string get_client(const token_properties& props)
    {
        if (auto listen_props = std::get_if<const listen_token_properties*>(&props))
        {
            return (*listen_props)->client;
        }
        return "";
    }

    template <typename StreamPtr>
    void log_unauthorized(const string& reason, const StreamPtr& stream)
    {
        auto& ctx = stream->request()->context;
        YLOG_CTX_GLOBAL(ctx, info) << "xtoken unauthorized: "
                                      "reason=\""
                                   << reason << "\"";
        ctx->custom_log_data["error"] = reason;
    }

    template <typename StreamPtr>
    void log_services_tokens(const detail::check_token_trace& trace, const StreamPtr& stream)
    {
        auto& custom_log_data = stream->request()->context->custom_log_data;
        custom_log_data["service"] = trace.service_names;
        custom_log_data["token"] = trace.token_names;
    }
};

struct multi_xtoken : public xtoken<xtoken_check_type::listen>
{
    template <typename StreamPtr, typename Handler>
    void operator()(
        const settings_ptr& /*settings*/,
        const StreamPtr& stream,
        const std::vector<string>& service_names,
        Handler&& handler)
    {
        auto tokens = parse_tokens(stream);
        if (tokens.empty())
        {
            handler(make_error(auth_error::no_credentials), multi_service_authorization{});
            return;
        }
        if (service_names.empty())
        {
            log_unauthorized("empty service list", stream);
            handler(make_error(auth_error::empty_service_list), multi_service_authorization{});
            return;
        }
        auto [check_result, check_trace, authorization] = check_tokens(tokens, service_names);
        if (!check_result)
        {
            log_unauthorized(check_result.error_reason, stream);
            handler(make_error(auth_error::multi_token_error), multi_service_authorization{});
            return;
        }
        log_services_tokens(check_trace, stream);
        handler(boost::system::error_code{}, authorization);
    }

private:
    std::tuple<operation::result, detail::check_token_trace, multi_service_authorization>
    check_tokens(const std::vector<string>& tokens, const std::vector<string>& services)
    {
        detail::string_set allowed_services;
        detail::check_token_trace check_trace;
        multi_service_authorization authorization;
        bool found_bad_token = false;
        for (auto&& token : tokens)
        {
            auto service = service_manager->find_service_by_listen_token(token);
            if (!service)
            {
                found_bad_token = true;
                continue;
            }
            auto ltoken_properties = find_token_properties(
                token, service_manager->environment(), service->listen_tokens);
            if (!ltoken_properties)
            {
                found_bad_token = true;
                continue;
            }
            authorization.push_back({ service->properties, ltoken_properties->client });
            check_trace.append(ltoken_properties->name, service->properties.name);
            allowed_services.insert(ltoken_properties->service);
            allowed_services.insert(
                ltoken_properties->allowed_services.begin(),
                ltoken_properties->allowed_services.end());
        }
        auto listen_not_allowed_services =
            allowed_services.missing_amongst(services.begin(), services.end());
        // Compatibility: return specific errors instead of service list for single token
        // authorization.
        if (found_bad_token && (listen_not_allowed_services.empty() || tokens.size() == 1))
        {
            return { "bad token", check_trace, {} };
        }
        if (single_item(listen_not_allowed_services) && tokens.size() == 1)
        {
            return { "forbidden service", check_trace, {} };
        }
        if (listen_not_allowed_services.size())
        {
            return { listen_not_allowed_services, check_trace, {} };
        }
        return { operation::result{}, check_trace, authorization };
    }

    bool single_item(const string& comma_separated_list)
    {
        return comma_separated_list.size() && comma_separated_list.find(',') == string::npos;
    }
};

}