#pragma once

#include "web/settings.h"
#include "web/find_deps.h"
#include "web/arguments/service_and_filter.h"
#include "web/auth/authorization.h"
#include <yxiva/core/operation_result.h>
#include <ymod_webserver/response.h>
#include <ymod_webserver/codes.h>
#include <ymod_blackbox/client.h>
#include <yplatform/find.h>
#include <boost/bind/protect.hpp>
#include <algorithm>
#include <functional>

namespace yxiva { namespace web {

typedef boost::shared_ptr<ymod_blackbox::client> bbclient_ptr;

namespace detail {

inline operation::result check_scopes(
    const ymod_blackbox::oauth_info_option& oauth_info,
    const std::set<string>& required)
{
    if (required.size())
    {
        auto& scopes = oauth_info.scopes;
        for (const auto& scope : required)
        {
            if (std::find(scopes.begin(), scopes.end(), scope) == scopes.end())
                return "invalid scopes";
        }
    }
    return operation::success;
}

operation::result fill_user_auth(
    user_authorization& /*auth*/,
    settings_ptr /*settings*/,
    const services_type& /*services*/,
    const ymod_blackbox::session_id_response& /*bb_response*/);

template <typename AuthHandler>
void on_auth_response(
    const ymod_blackbox::error& error,
    const ymod_blackbox::session_id_response& bb_response,
    settings_ptr settings,
    const services_type& services,
    AuthHandler&& handler)
{
    user_authorization auth;
    if (error || bb_response.status != ymod_blackbox::session_id_response::valid)
    {
        handler("bad response or status", auth);
        return;
    }

    auto resp_result = fill_user_auth(auth, settings, services, bb_response);
    handler(resp_result, auth);
}

inline const string& service_name(const string& name)
{
    return name;
}

inline const string& service_name(const service_with_filter& service)
{
    return service.name;
}

template <typename ServiceArg>
inline operation::result get_services(
    services_type& services,
    const ServiceArg& id,
    const service_manager& auth_service_manager)
{
    auto service = auth_service_manager.find_service_by_name(service_name(id));
    if (!service)
    {
        return "no service " + service_name(id);
    }
    services.push_back(std::move(service));
    return operation::success;
}

template <typename ServiceArg>
inline operation::result get_services(
    services_type& services,
    const std::vector<ServiceArg>& service_ids,
    const service_manager& auth_service_manager)
{
    services.reserve(service_ids.size());
    for (auto& id : service_ids)
    {
        auto res = get_services(services, id, auth_service_manager);
        if (!res) return res;
    }
    return operation::success;
}

struct resolve_config_with_oauth_impl
{
    template <typename AuthHandler>
    static void auth(
        bbclient_ptr bbclient,
        settings_ptr settings,
        const string& oauth_token,
        services_type&& services,
        const address& user_address,
        AuthHandler&& handler)
    {
        bbclient->async_oauth(
            oauth_token,
            user_address,
            [settings,
             services_ = std::move(services),
             handler_ =
                 std::forward<AuthHandler>(handler)](auto&& error, auto&& bb_response) mutable {
                on_auth_response(error, bb_response, settings, services_, handler_);
            });
    }
};

} // detail

template <typename ServiceArg, typename AuthHandler>
void resolve_config_with_oauth(
    bbclient_ptr bbclient,
    settings_ptr settings,
    const string& oauth_token,
    const ServiceArg& service_arg,
    const address& user_address,
    AuthHandler&& handler)
{
    services_type services;
    auto auth_service_manager = find_service_manager(settings->api.auth_service_manager);
    auto res = detail::get_services(services, service_arg, *auth_service_manager);
    if (!res)
    {
        handler(res, user_authorization{});
        return;
    }
    auto service_it = std::find_if(
        services.begin(), services.end(), [](const std::shared_ptr<const service_data>& svc) {
            return !svc->properties.is_passport;
        });
    if (service_it != services.end())
    {
        handler((*service_it)->properties.name + " not a passport service", user_authorization{});
        return;
    }
    detail::resolve_config_with_oauth_impl::auth(
        bbclient,
        settings,
        oauth_token,
        std::move(services),
        user_address,
        std::forward<AuthHandler>(handler));
};

} // web
} // yxiva