#pragma once

#include <yxiva/core/types.h>
#include <openssl/md5.h>
#include <boost/lexical_cast.hpp>
#include <ctime>

using boost::numeric_cast;

namespace yxiva { namespace web {

template <typename Items>
inline size_t total_representation_size(const Items& items)
{
    size_t size = items.size(); // commas
    for (auto& item : items)
    {
        size += item.size();
    }
    return size;
}

template <typename Items>
inline void append_to_representation(string& representation, const Items& items)
{
    for (auto& item : items)
    {
        representation += item;
        representation += ',';
    }
}

template <typename Services, typename Uids, typename Topics>
inline string compose_sign_data(const Services& services, const Uids& uids, const Topics& topics)
{
    size_t sign_data_size = 0;
    sign_data_size += total_representation_size(uids);
    sign_data_size += total_representation_size(topics);
    sign_data_size += total_representation_size(services);

    string sign_data;
    sign_data.reserve(sign_data_size);
    append_to_representation(sign_data, topics);
    if (sign_data.size())
    {
        sign_data.back() = ':';
    }
    append_to_representation(sign_data, services);
    append_to_representation(sign_data, uids);

    return sign_data;
}

inline string make_secure_sign(const string& data, time_t ts, const string& secret)
{
    MD5_CTX md5handler;
    unsigned char md5buffer[16];

    MD5_Init(&md5handler);
    MD5_Update(&md5handler, data.c_str(), data.length());
    if (!secret.empty())
    {
        MD5_Update(&md5handler, secret.c_str(), secret.length());
    }
    if (ts)
    {
        char buf[40];
        int len = sprintf(&buf[0], "%llu", numeric_cast<unsigned long long>(ts));
        MD5_Update(&md5handler, &buf[0], len);
    }
    MD5_Final(md5buffer, &md5handler);

    char alpha[16] = { '0', '1', '2', '3', '4', '5', '6', '7',
                       '8', '9', 'a', 'b', 'c', 'd', 'e', 'f' };
    unsigned char c;
    string md5digest;
    md5digest.reserve(32);

    for (int i = 0; i < 16; ++i)
    {
        c = (md5buffer[i] & 0xf0) >> 4;
        md5digest.push_back(alpha[c]);
        c = (md5buffer[i] & 0xf);
        md5digest.push_back(alpha[c]);
    }
    return md5digest;
}

struct services_uids_topic_data
{
    explicit services_uids_topic_data(
        const std::vector<string>& services,
        const std::vector<string>& uids,
        const string& topic)
        : services(services), uids(uids), topic(topic)
    {
    }
    const std::vector<string>& services;
    const std::vector<string>& uids;
    const string& topic;
};

struct services_uids_topics_data
{
    explicit services_uids_topics_data(
        const std::vector<string>& services,
        const std::vector<string>& uids,
        const std::vector<string>& topics)
        : services(services), uids(uids), topics(topics)
    {
    }
    const std::vector<string>& services;
    const std::vector<string>& uids;
    const std::vector<string>& topics;
};

struct uid_data
{
    explicit uid_data(const string& uid) : uid(uid)
    {
    }
    const string& uid;
};

inline string make_secure_sign(services_uids_topic_data data, time_t ts, const string& secret)
{
    string composed_data;
    if (data.topic.size())
    {
        composed_data =
            compose_sign_data(data.services, data.uids, std::vector<string>{ data.topic });
    }
    else
    {
        composed_data = compose_sign_data(data.services, data.uids, std::vector<string>{});
    }
    return make_secure_sign(composed_data, ts, secret);
}

inline string make_secure_sign(services_uids_topics_data data, time_t ts, const string& secret)
{
    auto composed_data = compose_sign_data(data.services, data.uids, data.topics);
    return make_secure_sign(composed_data, ts, secret);
}

inline string make_secure_sign(uid_data data, time_t ts, const string& secret)
{
    return make_secure_sign(data.uid, ts, secret);
}

}}
