#include "user.h"

#include <yxiva/core/authorizer.h>
#include <yplatform/util/sstream.h>
#include <yplatform/util/safe_call.h>

namespace yxiva { namespace web {
using namespace auth_error;

user_info_future_t auth_user(request_ptr req)
{
    string uid = req->url.param_value("uid", "");
    if (!uid.empty())
    {
        return find_processor()->authorizer()->authenticate(
            req->context,
            { req->context->remote_address, req->context->remote_port },
            user_id(uid));
    }
    user_info_promise_t aresp;
    aresp.set_exception(std::runtime_error("no user id"));
    return aresp;
}

namespace {

void log_multiinfo(task_context_ptr ctx, const auth_info& auth)
{
    std::stringstream ss;
    for (auto it = auth.users.begin(); it != auth.users.end(); ++it)
    {
        ss << " (" << it->uid << ")";
    }
    YLOG_CTX_GLOBAL(ctx, info) << "bb info" << ss.str();
}

}

void handle_auth_call(
    task_context_ptr ctx,
    const boost::system::error_code& err,
    ymod_http_client::response response,
    const auth_callback_t& cb) noexcept
{
    static const string WHERE = "user auth";
    static const auth_info EMPTY_INFO;
    try
    {
        if (err)
        {
            safe_call(ctx, WHERE, cb, err, EMPTY_INFO);
            return;
        }
        if (response.status == 200)
        {
            json_value value;
            if (auto error = value.parse(response.body))
            {
                YLOG_CTX_GLOBAL(ctx, error) << "invalid response from the auth host: " << *error;
                safe_call(ctx, WHERE, cb, make_error(request_failed), EMPTY_INFO);
                return;
            }

            auto error = value["error"];
            if (!error.is_null())
            {
                YLOG_CTX_GLOBAL(ctx, info) << "auth error response='" << error.stringify() << "'";
                safe_call(ctx, WHERE, cb, make_error(authorization_error), EMPTY_INFO);
                return;
            }

            auto&& json_account = value["check_cookies"];
            if (json_account.is_null())
            {
                YLOG_CTX_GLOBAL(ctx, info)
                    << "no account info response='" << value.stringify() << "'";
                safe_call(ctx, WHERE, cb, make_error(no_account_info), EMPTY_INFO);
                return;
            }

            // convert to user_info
            auth_info result;
            result.bb_connection_id = json_get<string>(json_account, "bbConnectionId", "");
            user_info account_info;
            account_info.uid = json_get<string>(json_account, "uid", "");
            account_info.timezone_offset = json_get<int>(json_account, "offset", 14400);
            result.users.push_back(account_info);
            if (static_cast<string>(result.users.front().uid).empty())
            {
                safe_call(ctx, WHERE, cb, make_error(empty_user_info), EMPTY_INFO);
                return;
            }

            auto&& child_uids = json_account["childUids"];
            if (child_uids.is_array())
            {
                for (auto&& child_uid : child_uids.array_items())
                {
                    user_info child_ui;
                    child_ui.uid = json_get<string>(child_uid, "");
                    child_ui.timezone_offset = account_info.timezone_offset;
                    if (!static_cast<string>(child_ui.uid).empty())
                    {
                        result.users.push_back(child_ui);
                    }
                }
            }

            log_multiinfo(ctx, result);
            safe_call(ctx, WHERE, cb, make_error(success), result);
        }
        else
        {
            YLOG_CTX_GLOBAL(ctx, info) << "http error on auth: code " << response.status
                                       << ", body \"" << response.body << "\"";
            safe_call(ctx, WHERE, cb, make_error(bad_host_response), EMPTY_INFO);
        }
    }
    catch (const std::exception& e)
    {
        YLOG_CTX_GLOBAL(ctx, info) << "exception on auth: " << e.what();
        safe_call(ctx, WHERE, cb, make_error(unknown_error), EMPTY_INFO);
    }
    catch (...)
    {
        YLOG_CTX_GLOBAL(ctx, info) << "unknown exception on auth";
        safe_call(ctx, WHERE, cb, make_error(unknown_error), EMPTY_INFO);
    }
}

void auth_user_cookies(
    const std::shared_ptr<yhttp::cluster_client>& http_client,
    task_context_ptr ctx,
    const string& cookie,
    const string& auth_domain,
    const address& user_address,
    auth_callback_t cb)
{
    namespace p = std::placeholders;
    string headers;
    yplatform::sstream(headers) << "Cookie: " << cookie << "\r\n"
                                << "X-Original-Host: " << auth_domain << "\r\n"
                                << "X-Real-Ip: " << user_address.ip << "\r\n"
                                << "X-Real-Port: " << user_address.port << "\r\n";

    http_client->async_run(
        ctx,
        ymod_http_client::request::GET("/check_cookies", headers),
        std::bind(&handle_auth_call, ctx, p::_1, p::_2, std::move(cb)));
}

void auth_user_cookies(
    task_context_ptr ctx,
    const string& cookie,
    const string& auth_domain,
    const address& user_address,
    auth_callback_t cb)
{
    auth_user_cookies(
        yplatform::find<yhttp::cluster_client, std::shared_ptr>("akita_client"),
        std::move(ctx),
        cookie,
        auth_domain,
        user_address,
        std::move(cb));
}

}}
