#pragma once

#include "common.h"
#include <yxiva/core/services/names.h>

namespace yxiva { namespace web { namespace backapi {

namespace hacks {
void mail_reset_operation_unsupported(const string& service, string& operation)
{
    if (service == "mail" && operation != "insert")
    {
        operation = "unsupported";
    }
}
}

namespace {
service_name extract_service_name(ymod_webserver::request& req)
{
    return service_name(
        req.url.path.size() > 1 ? req.url.path[1] :
                                  static_cast<string>(services::hardcoded_names::MAIL));
}
}

void notify(settings_ptr /*settings*/, const http_stream_ptr& stream)
{
    static const string decoder_name = "msgpack";

    auto req = stream->request();

    message msg;
    service_name service = extract_service_name(*req);
    auto& custom_log_data = stream->request()->context->custom_log_data;
    custom_log_data["service"] = service;

    try
    {
        if (auto decoder = find_processor()->decoder(decoder_name))
        {
            decoder->decode(req, msg);
        }
        else
        {
            WEB_RESPONSE_LOG_G(info, stream, not_found, "no such decoder " + decoder_name);
            return;
        }
    }
    catch (const std::exception& e)
    {
        WEB_RESPONSE_LOG_G(
            info, stream, bad_request, string("failed to decode message: ") + e.what());
        return;
    }

    hacks::mail_reset_operation_unsupported(msg.service, msg.operation);
    catalogue_ptr catalogue = find_processor()->catalogue();
    if (!catalogue)
    {
        WEB_RESPONSE_LOG_G(error, stream, internal_server_error, "no catalogue");
        return;
    }

    if (msg.uid.empty())
    {
        WEB_RESPONSE_LOG_G(error, stream, bad_request, "no uid specified");
        return;
    }

    auto subscription_id = req->url.param_value("subscription_id", "");
    std::vector<subscriber_ptr> subs;
    if (subscription_id.empty())
    {
        subs = catalogue->find_by_uid_service(msg.uid, service_name(msg.service));
        if (subs.empty())
        {
            WEB_RESPONSE_LOG_G(
                info,
                stream,
                reset_content,
                "no subs for user '" + msg.uid + "' in service '" + msg.service + "'");
            find_xivaws_log()->log_notification_skip(req->context, msg, "no subs");
            return;
        }
    }
    else
    {
        auto sub = catalogue->find_by_id(msg.uid, service_name(msg.service), subscription_id);
        if (!sub)
        {
            WEB_RESPONSE_LOG_G(
                info,
                stream,
                reset_content,
                "no sub with subscription_id '" + subscription_id + "' for user '" + msg.uid +
                    "' in service '" + msg.service + "'");
            find_xivaws_log()->log_notification_skip(
                req->context, msg, "no sub by subscription_id");
            return;
        }
        else
        {
            subs.push_back(sub);
        }
    }

    string req_context = (req->context ? req->context->uniq_id() : "");
    for (auto& sub : subs)
    {
        find_xivaws_log()->log_notification(req->context, subscription_id, *sub, msg);
        sub->notify(
            user_info(user_id(msg.uid)),
            msg,
            req_context + ":" + msg.transit_id + ":" + sub->client_id() + ":" + msg.service);
    }

    WEB_RESPONSE(stream, ok, "OK");
}
}}}
