#include "bb.h"
#include <yxiva/core/callbacks.h>
#include <yxiva/core/types.h>
#include <yplatform/find.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace yxiva { namespace web { namespace extapi {
namespace {

template <typename Handler>
inline void just_wait(const time_duration& d, const http_stream_ptr& stream, Handler&& h)
{
    auto timer = stream->make_timer();
    timer->expires_from_now(d);
    timer->async_wait(std::forward<Handler>(h));
}

// TODO: put this in settings?
static const string BB_LOGIN_SERVICE = "bass";

struct bb_login_coro
{
    http_stream_ptr stream;
    settings_ptr settings;
    string user;
    string yandexuid;
    string bb_connection_id;

    struct subscribe_params
    {
        string client;
        string session_key;
        string id;
        string filter;
        string callback;
        string extra_data;
        time_t init_time = 0;
    };

    using yield_context = yplatform::yield_context<bb_login_coro>;
    void operator()(
        yield_context yield_ctx,
        const boost::system::error_code& err = {},
        yhttp::response response = {})
    {
        string error_text;
        reenter(yield_ctx)
        {
            yield list_yandexuid_subs(yield_ctx);
            if (err || response.status != 200)
            {
                respond_error(err, response, "hub list failed");
                return;
            }
            subs_ = parse_subs(response.body, error_text);
            if (error_text.size())
            {
                respond_error(error_text);
                return;
            }
            sub_to_copy_ = find_sub_to_copy();
            if (!sub_to_copy_)
            {
                respond_no_subs();
                return;
            }
            yield copy_sub(yield_ctx);
            if (err || response.status != 200)
            {
                respond_error(err, response, "hub subscribe failed");
                return;
            }
            stream->result(http_codes::ok);
        }
    }

    void list_yandexuid_subs(yield_context yield_ctx)
    {
        find_hubrpc()->async_get(
            stream->ctx(),
            yandexuid,
            "/list_json",
            { { "uid", yandexuid }, { "service", BB_LOGIN_SERVICE }, { "priority", "high" } },
            yield_ctx);
    }

    std::vector<subscribe_params> parse_subs(const string& body, string& parse_error)
    {
        parse_error = {};
        json_value list;
        if (auto error = list.parse(body, json_type::tarray))
        {
            parse_error = "hub list result parse error: " + *error;
            return {};
        }
        try
        {
            std::vector<subscribe_params> subs;
            subs.reserve(list.size());
            for (auto&& list_item : list.array_items())
            {
                subscribe_params sub;
                sub.client = list_item["client"].to_string();
                sub.session_key = list_item["session_key"].to_string();
                sub.id = list_item["id"].to_string();
                sub.filter = list_item["filter"].to_string();
                sub.callback = list_item["url"].to_string();
                sub.extra_data = list_item["extra_data"].to_string();
                sub.init_time = list_item["init_time"].to_uint64();
                subs.push_back(std::move(sub));
            }
            return subs;
        }
        catch (const std::exception& e)
        {
            parse_error = string("invalid json field: ") + e.what();
            return {};
        }
    }

    std::optional<subscribe_params> find_sub_to_copy()
    {
        static const string yabrowser_client = "YandexBrowser";
        using callback_uri::is_webpush_uri;
        std::optional<subscribe_params> ret;
        for (auto&& sub : subs_)
        {
            if (is_webpush_uri(sub.callback) && sub.client == yabrowser_client &&
                (!ret || ret->init_time < sub.init_time))
            {
                ret = sub;
            }
        }
        return ret;
    }

    void copy_sub(const yield_context& yield_ctx)
    {
        string uidservice = user + BB_LOGIN_SERVICE;
        find_hubrpc()->async_post(
            stream->ctx(),
            user,
            "/subscribe",
            { { "uid", user },
              { "service", BB_LOGIN_SERVICE },
              { "uidservice", uidservice },
              { "client", sub_to_copy_->client },
              { "session_key", sub_to_copy_->session_key },
              { "ttl", settings->api.hub.subscribe_bb_login_ttl },
              { "id", sub_to_copy_->id },
              { "account", user },
              { "bb_connection_id", bb_connection_id },
              { "priority", "high" } },
            yhttp::form_encode({ { "filter", sub_to_copy_->filter },
                                 { "callback", sub_to_copy_->callback },
                                 { "extra_data", sub_to_copy_->extra_data } }),
            yield_ctx);
    }

    void respond_error(
        const boost::system::error_code& err,
        const yhttp::response& response,
        const string& reason)
    {
        if (err)
        {
            YLOG_CTX_GLOBAL(stream->ctx(), error) << reason << ": error=" << err.message();
        }
        else
        {
            YLOG_CTX_GLOBAL(stream->ctx(), info)
                << reason << ": code=" << response.status << " error=" << response.body;
        }
        stream->result(http_codes::internal_server_error);
    }

    void respond_error(const string& error)
    {
        YLOG_CTX_GLOBAL(stream->ctx(), error) << error;
        stream->result(http_codes::internal_server_error);
    }

    void respond_no_subs()
    {
        YLOG_CTX_GLOBAL(stream->ctx(), info) << "no subs to copy for " << yandexuid;
        stream->result(http_codes::ok);
    }

    std::vector<subscribe_params> subs_ = {};
    std::optional<subscribe_params> sub_to_copy_ = {};
};

}

void bb_login(
    http_stream_ptr stream,
    settings_ptr settings,
    const service_authorization& auth,
    const string& /*service*/,
    const string& user,
    const string& yandexuid,
    const string& bb_connection_id)
{
    if (auth.service.name != BB_LOGIN_SERVICE)
    {
        send_unauthorized(stream, "bad token");
        return;
    }
    if (settings->api.hub.subscribe_bb_login_ttl > 0)
    {
        yplatform::spawn(std::make_shared<bb_login_coro>(
            bb_login_coro{ stream, settings, user, yandexuid, bb_connection_id }));
    }
    else
    {
        using boost::system::error_code;
        just_wait(milliseconds(2), stream, [stream](const error_code&) {
            stream->result(http_codes::ok);
        });
    }
}

void bb_logout(
    http_stream_ptr stream,
    settings_ptr /*settings*/,
    const service_authorization& /*auth*/,
    const string& /*string*/,
    const string& /*user*/,
    const string& /*yandexuid*/)
{
    // TODO: implement.
    using boost::system::error_code;
    just_wait(
        milliseconds(2), stream, [stream](const error_code&) { stream->result(http_codes::ok); });
}

}}}
#include <yplatform/unyield.h>
