#include "mobile_push.h"

#include <yxiva/core/platforms.h>
#include <yplatform/find.h>
#include <functional>

namespace yxiva { namespace web { namespace extapi {

namespace {
void handle_proxy_call(
    const boost::system::error_code& ec,
    yhttp::response resp,
    http_stream_ptr stream,
    const string& platform)
{
    stream->request()->context->profilers.pop();
    if (ec)
    {
        YLOG_CTX_GLOBAL(stream->ctx(), info)
            << "mobile push platform " << platform << " failed, exception=" << ec.message();
        stream->result(http_codes::internal_server_error, ec.message());
        return;
    }
    YLOG_CTX_GLOBAL(stream->ctx(), info)
        << "mobile push platform " << platform << " result: " << resp.status << " " << resp.body;
    stream->result(static_cast<ymod_webserver::codes::code>(resp.status), resp.body);
}
}

void mobile_push(
    http_stream_ptr stream,
    const service_authorization&,
    const string& platform,
    bool batch = false)
{
    string url;
    if (platform == yxiva::platform::FCM)
    {
        url = batch ? "/batch_push/gcm" : "/push/gcm";
    }
    else if (platform == yxiva::platform::APNS)
    {
        url = "/push/apns";
    }
    else if (platform == yxiva::platform::MPNS)
    {
        url = "/push/mpns";
    }
    else if (platform == yxiva::platform::WNS)
    {
        url = "/push/wns";
    }
    else
    {
        stream->result(http_codes::internal_server_error);
        YLOG_CTX_GLOBAL(stream->ctx(), error) << "platform " << platform << " is not supported";
        return;
    }
    if (batch && platform != yxiva::platform::FCM)
    {
        stream->result(http_codes::internal_server_error);
        YLOG_CTX_GLOBAL(stream->ctx(), error)
            << "batching for platform " << platform << " is not supported";
        return;
    }

    auto req = stream->request();
    auto& ctx = req->context;
    auto http_client = yplatform::find<yhttp::cluster_client>("ext_api_client");

    sstream(url, 50) << "?app=" << url_encode(req->url.param_value("app_name"))
                     << "&ttl=" << url_encode(req->url.param_value("ttl"));
    string body;
    sstream body_stream(body, req->raw_body.size());
    body_stream << "payload=" << url_encode(req->url.param_value("payload"));
    if (batch)
    {
        body_stream << "&tokens=" << url_encode(req->url.param_value("push_tokens"));
    }
    else
    {
        body_stream << "&token=" << url_encode(req->url.param_value("push_token"));
    }
    for (auto& param : req->url.params)
    {
        if (!param.first.compare(0, 2, "x-"))
        {
            body_stream << "&" << param.first << "=" << url_encode(param.second);
        }
    }

    namespace _p = std::placeholders;
    ctx->profilers.push("mob::push_" + platform + (batch ? "_batch" : ""));
    http_client->async_run(
        ctx,
        yhttp::request::POST(url, std::move(body)),
        std::bind(handle_proxy_call, _p::_1, _p::_2, stream, platform));
}

void mobile_push_apns(http_stream_ptr stream, settings_ptr, const service_authorization& auth)
{
    mobile_push(stream, auth, yxiva::platform::APNS);
}

void mobile_push_fcm(http_stream_ptr stream, settings_ptr, const service_authorization& auth)
{
    mobile_push(stream, auth, yxiva::platform::FCM);
}

void mobile_push_mpns(http_stream_ptr stream, settings_ptr, const service_authorization& auth)
{
    mobile_push(stream, auth, yxiva::platform::MPNS);
}

void mobile_push_wns(http_stream_ptr stream, settings_ptr, const service_authorization& auth)
{
    mobile_push(stream, auth, yxiva::platform::WNS);
}

void mobile_batch_push_fcm(http_stream_ptr stream, settings_ptr, const service_authorization& auth)
{
    mobile_push(stream, auth, yxiva::platform::FCM, true);
}

}}}
