#include "address.h"

namespace yxiva { namespace rfc2822 {

using std::list;
typedef string::size_type size_type;

bool atomchar(char c)
{
    return !(
        (c > 0 && c <= ' ') || c == '(' || c == '"' || c == ')' || c == ',' || c == '.' ||
        c == ':' || c == ';' || c == '<' || c == '>' || c == '[' || c == ']' || c == '@');
}

string quote(const string& src)
{
    bool quoted = false;
    string dst, space;
    dst.reserve(src.size() + 2);
    size_type i = 0;

    while (i < src.size() && src[i] == ' ')
        dst += src[i++];
    dst += '\"';

    for (; i < src.size(); ++i)
    {
        if (src[i] == ' ') space += src[i];
        else
        {
            dst += space;
            space.clear();
            if (src[i] == '\\' || src[i] == '\"')
            {
                dst += '\\';
                quoted = true;
            }
            else if (!atomchar(src[i]) && src[i] != '.')
                quoted = true;
            dst += src[i];
        }
    }
    dst += '\"';
    dst += space;
    if (quoted) return dst;
    else
        return src;
}

string quote(const list<string>& src)
{
    string dst;
    for (list<string>::const_iterator i = src.begin(); i != src.end(); ++i)
    {
        if ((*i)[0] == '(') dst += " " + *i;
        else
            dst += quote(*i);
    }
    return dst;
}

string paste(const list<string>& src)
{
    string dst;
    for (list<string>::const_iterator i = src.begin(); i != src.end(); ++i)
    {
        if ((*i)[0] == '(') dst += " " + *i;
        else
            dst += *i;
    }
    return dst;
}

string join_address(const string& local, const string& domain)
{
    if (domain.empty()) return quote(local);
    return quote(local) + "@" + domain;
}

bool unquote_pair(const string& src, size_type& pos, string& dst)
{
    if (pos + 1 >= src.size() || src[pos] != '\\') return false;
    pos++;
    dst += src[pos++];
    return true;
}

bool quoted_pair(const string& src, size_type& pos, string& dst)
{
    if (pos + 1 >= src.size() || src[pos] != '\\') return false;
    dst += src[pos++];
    dst += src[pos++];
    return true;
}

bool fws(const string& src, size_type& pos, string& dst)
{
    size_type tmppos = pos;
    while (
        tmppos < src.size() &&
        (src[tmppos] == ' ' || src[tmppos] == '\t' || src[tmppos] == '\r' || src[tmppos] == '\n'))
        tmppos++;
    if (tmppos == pos) return false;
    pos = tmppos;
    if (*dst.rbegin() != ' ') dst += ' ';
    return true;
}

bool comment(const string& src, size_type& pos, string& dst)
{
    if (pos >= src.size() || src[pos] != '(') return false;
    size_type tmppos = pos;
    string tmpdst;
    tmpdst += src[tmppos++];
    while (tmppos < src.size() && src[tmppos] != ')')
    {
        if (!quoted_pair(src, tmppos, tmpdst) && !fws(src, tmppos, tmpdst) &&
            !comment(src, tmppos, tmpdst))
            tmpdst += src[tmppos++];
    }
    if (src[tmppos] != ')') return false;
    tmpdst += src[tmppos++];
    dst += tmpdst;
    pos = tmppos;
    return true;
}

bool quoted_string(const string& src, size_type& pos, string& dst)
{
    if (pos >= src.size() || src[pos] != '\"') return false;
    size_type tmppos = pos + 1;
    string tmpdst;
    while (tmppos < src.size() && src[tmppos] != '\"')
    {
        if (!unquote_pair(src, tmppos, tmpdst) && !fws(src, tmppos, tmpdst))
            tmpdst += src[tmppos++];
    }
    if (src[tmppos] != '\"') return false;
    dst += tmpdst;
    pos = tmppos + 1;
    return true;
}

bool atom(const string& src, size_type& pos, string& dst)
{
    size_type tmppos = pos;
    string tmpdst;
    while (tmppos < src.size() && atomchar(src[tmppos]))
        tmpdst += src[tmppos++];
    if (src[tmppos] == '.') return false;
    if (tmppos == pos || (tmppos < src.size() && src[tmppos] == '.')) return false;
    pos = tmppos;
    dst += tmpdst;
    return true;
}

bool dotatom(const string& src, size_type& pos, string& dst)
{
    size_type tmppos = pos;
    string tmpdst;
    while (tmppos < src.size() && (atomchar(src[tmppos]) || src[tmppos] == '.'))
        tmpdst += src[tmppos++];
    if (tmppos == pos) return false;
    pos = tmppos;
    dst += tmpdst;
    return true;
}

bool xatom(const string& src, size_type& pos, string& dst)
{
    string tmpdst, space;
    size_type tmppos = pos;
    if (!dotatom(src, tmppos, tmpdst)) return false;
    size_type xpos = tmppos;
    fws(src, xpos, space);
    if (src[xpos] == '@') return false;
    pos = tmppos;
    dst += tmpdst;
    return true;
}

bool display_name(const string& src, size_type& pos, list<string>& dst)
{
    bool have_result = false;
    list<string> tmpdst;
    string data, commentdata;
    size_type tmppos = pos;
    while (tmppos < src.size())
    {
        string space, tmpdata;
        fws(src, tmppos, space);
        if (!xatom(src, tmppos, tmpdata) && !quoted_string(src, tmppos, tmpdata))
        {
            if (comment(src, tmppos, commentdata))
            {
                if (!data.empty()) tmpdst.push_back(data);
                fws(src, tmppos, commentdata);
                tmpdst.push_back(commentdata);
                data.clear();
                commentdata.clear();
            }
            else if (src[tmppos] == '\"' || src[tmppos] == '(')
                data += space + src[tmppos++];
            else
                break;
        }
        else if (tmpdata.empty())
        {
            data += space;
            have_result = true;
        }
        else
            data += space + tmpdata;
    }
    if (!data.empty()) tmpdst.push_back(data);
    if (tmpdst.empty() && !have_result) return false;
    pos = tmppos;
    dst.swap(tmpdst);
    return true;
}

bool domain_literal(const string& src, size_type& pos, string& dst)
{
    if (pos >= src.size() || src[pos] != '[') return false;
    size_type tmppos = pos;
    string tmpdst;
    tmpdst += src[tmppos++];
    while (tmppos < src.size() && src[tmppos] != ']')
    {
        if (src[tmppos] == '[') return false;
        if (!quoted_pair(src, tmppos, tmpdst) && !fws(src, tmppos, tmpdst)) tmpdst += src[tmppos++];
    }
    if (src[tmppos] != ']') return false;
    tmpdst += src[tmppos++];
    dst += tmpdst;
    pos = tmppos;
    return true;
}

bool addrspec(const string& src, size_type& pos, string& local, string& domain)
{
    string tmplocal, tmpdomain, space;
    string::size_type tmppos = pos;
    if (!dotatom(src, tmppos, tmplocal) && !quoted_string(src, tmppos, tmplocal)) return false;
    fws(src, tmppos, space);
    if (src[tmppos] != '@') return false;
    tmppos++;
    fws(src, tmppos, space);
    if (!dotatom(src, tmppos, tmpdomain) && !domain_literal(src, tmppos, tmpdomain)) return false;
    pos = tmppos;
    local += tmplocal;
    domain += tmpdomain;
    return true;
}

bool addrspec(const string& src, size_type& pos, string& dst)
{
    string local, domain;
    if (!addrspec(src, pos, local, domain)) return false;
    dst += join_address(local, domain);
    return true;
}

bool angleaddr(const string& src, size_type& pos, string& local, string& domain)
{
    string tmplocal, tmpdomain;
    size_type tmppos = pos;
    if (tmppos >= src.size() || src[tmppos] != '<') return false;
    tmppos++;
    if (!addrspec(src, tmppos, tmplocal, tmpdomain) && !dotatom(src, tmppos, tmplocal))
        return false;
    if (tmppos >= src.size() || src[tmppos] != '>') return false;
    tmppos++;
    pos = tmppos;
    local += tmplocal;
    domain += tmpdomain;
    return true;
}

bool angleaddr(const string& src, size_type pos, string& dst)
{
    string local, domain;
    if (!angleaddr(src, pos, local, domain)) return false;
    dst += join_address(local, domain);
    return true;
}

bool address(
    const string& src,
    size_type& pos,
    list<string>& display,
    string& local,
    string& domain)
{
    list<string> tmpdisplay;
    string tmplocal, tmpdomain, space;
    size_type tmppos = pos;
    bool have_name = display_name(src, tmppos, tmpdisplay);
    fws(src, tmppos, space);
    if (!angleaddr(src, tmppos, tmplocal, tmpdomain) && !addrspec(src, tmppos, tmplocal, tmpdomain))
        if (!have_name) return addrspec(src, pos, local, domain);
    pos = tmppos;
    display.swap(tmpdisplay);
    local = tmplocal;
    domain = tmpdomain;
    return true;
}

bool address(const string& src, size_type& pos, string& display, string& addr)
{
    list<string> tmpdisplay;
    string local, domain;
    if (!address(src, pos, tmpdisplay, local, domain)) return false;
    display += quote(tmpdisplay);
    addr = join_address(local, domain);
    return true;
}

address_iterator& address_iterator::operator++(void)
{
    list<string> tmpdisplay;
    string space, tmplocal, tmpdomain;
    rfc2822::fws(src_, pos_, space);
    while (pos_ < src_.size() && (src_[pos_] == ',' || src_[pos_] == ';'))
    {
        pos_++;
        rfc2822::fws(src_, pos_, space);
    }
    if (!rfc2822::address(src_, pos_, tmpdisplay, tmplocal, tmpdomain))
    {
        if (pos_ < src_.size())
        {
            string text = src_.substr(pos_, src_.size() - pos_);
            throw invalid_address(text);
        }
        pos_ = string::npos;
    }
    else
    {
        local_ = tmplocal;
        domain_ = tmpdomain;
        display_ = quote(tmpdisplay);
        pretty_ = paste(tmpdisplay);
        address_ = join_address(local_, domain_);
    }
    return *this;
}

const string address_iterator::empty;
}}
