#include "daria.h"

#include <boost/lexical_cast.hpp>

#include "web/formatters/address.h"
#include "web/formatters/timeutils.h"
#include "web/formatters/address.h"
#include <yxiva/core/quote_xml.h>
#include <yxiva/core/map_utils.h>
#include <yxiva/core/json.h>
#include <yxiva/core/parameter_names.h>
#include <yxiva/core/services/names.h>
#include <yplatform/hash/md5.h>

namespace yxiva { namespace formatters {
namespace {

void format_address(std::ostream& out, const string& addr, const string& root)
{
    try
    {
        rfc2822::address_iterator addr_it(addr), addr_end;
        std::string name = addr_it.pretty();
        std::string email = addr_it.address();
        if (name.empty()) name = email;
        while (addr_it != addr_end)
        {
            out << "  <" << root << ">"
                << "    <name>";

            // XXX: security by obscurity. need to rework.
            auto md5sum = yplatform::md5(
                "6XatLBy8MBA7GshwTNtXqL6DNcJvKxBS", email, "SrupGzRbffG3ZCkn60gdsHu2MAPV6fPE");

            quoteXML(name, out);
            out << "</name>"
                << "    <email ref=\"" << md5sum << "\">";
            quoteXML(email, out);
            out << "</email>"
                << "  </" << root << ">";
            addr_it++;
        }
    }
    catch (const std::exception& e)
    {
        L_(error) << "daria_message_format address exception: " << e.what();
    }
    catch (...)
    {
        L_(error) << "daria_message_format address exception: unknown";
    }
}

using std::string;
using std::fixed;

const int kbSize = 1024;
const char* prefixes[] = { "байт", "КБ", "МБ", "ГБ" };
const int prefixesCount = 4;

string size2humanreadable(long size)
{
    int prefixNumber = 0;
    long rangeLow = 1;
    long rangeHigh = kbSize;
    while ((size > rangeHigh) && (prefixNumber < prefixesCount))
    {
        ++prefixNumber;
        rangeLow = rangeHigh;
        rangeHigh *= kbSize;
    }

    const double rangedSize = static_cast<double>(size) * 1.0 / static_cast<double>(rangeLow);
    std::ostringstream ss;
    if (rangedSize > 10.0)
    {
        ss.precision(0);
        ss << fixed << rangedSize;
    }
    else
    {
        ss.precision(1);
        ss << fixed << rangedSize;
    }

    ss << " " << prefixes[prefixNumber];

    string result = ss.str();
    std::size_t pos = result.find('.');

    if (pos != string::npos) result[pos] = ',';

    return result;
}

// received_date:28.05.2010 19:17:26

void format_date(std::ostream& out, const std::string& date, int timezoneOffset)
{
    out << "  <date>"
        << "    <full>" << fromStrDateToFullDate(date.c_str(), timezoneOffset) << "</full>"
        << "    <short>" << fromStrDateToShortDate(date.c_str(), timezoneOffset) << "</short>"
        << "    <iso>" << fromStrDateToISODate(date.c_str(), timezoneOffset) << "</iso>"
        << "  </date>";
}

string build_mail_message_view_daria(const std::map<string, string>& proplist, int timezoneOffset)
{
    int is_mixed = atoi(find_map_value(proplist, "is_mixed").c_str());

    std::ostringstream xml_stream;
    xml_stream << "<message id=\"" << find_map_value(proplist, "mid") << "\">"
               << "  <thread id=\"" << find_map_value(proplist, "thread_id") << "\"/>"
               << "  <folder id=\"" << find_map_value(proplist, "fid") << "\"/>"
               << "  <subject>";
    quoteXML(find_map_value(proplist, "hdr_subject"), xml_stream);
    xml_stream << "</subject>  <firstline>";
    quoteXML(find_map_value(proplist, "firstline"), xml_stream);
    xml_stream << "</firstline>";
    format_address(xml_stream, find_map_value(proplist, "hdr_from"), "from");
    format_address(xml_stream, find_map_value(proplist, "hdr_to"), "to");

    // parse comma-separated list of labels
    std::string lids = find_map_value(proplist, "lid");
    // comma-separsted list of labels
    if (!lids.empty())
    {
        xml_stream << "  <labels>";
        std::string lid;
        for (size_t i = 0; i < lids.size(); ++i)
        {
            char c = lids[i];
            if (c == ',')
            {
                if (!lid.empty())
                {
                    xml_stream << "    <label id=\"" << lid << "\"/>";
                    lid = "";
                }
            }
            else
            {
                lid.push_back(c);
            }
        }
        if (!lid.empty()) xml_stream << "    <label id=\"" << lid << "\"/>";
        xml_stream << "  </labels>";
    }
    else
    {
        xml_stream << "  <labels/>";
    }

    int sz = atoi(find_map_value(proplist, "sz").c_str());
    std::string size = size2humanreadable(sz);
    xml_stream << "  <size>" << size << "</size>";

    format_date(xml_stream, find_map_value(proplist, "received_date"), timezoneOffset);

    if (find_map_value(proplist, "hdr_status") == "New")
    {
        xml_stream << "  <new/>";
    }

    if (is_mixed & 0x01)
    {
        xml_stream << "  <flags><attachment/></flags>";
    }

    xml_stream << "</message>";
    return xml_stream.str();
}

string generate_data(
    const string& operation,
    const std::map<string, string>& proplist,
    int tzOffset)
{
    if (operation == "insert")
    {
        return build_mail_message_view_daria(proplist, tzOffset);
    }
    return "";
}
}

void daria::operator()(
    const user_info& info,
    const message& msg,
    const string& context_id,
    std::ostream& out) const
{
    json_value daria_msg;
    daria_msg["service"] = static_cast<string>(msg.service);
    daria_msg["lcn"] = msg.lcn;
    daria_msg["operation"] = msg.operation;
    daria_msg["connection_id"] = msg.session_key;
    daria_msg["method_id"] = find_map_value(msg.data, "method_id");
    daria_msg["server_notify_id"] = context_id;

    if (msg.service == static_cast<std::string>(services::hardcoded_names::MAIL))
    {
        daria_msg["data"] = generate_data(msg.operation, msg.data, info.timezone_offset);
        daria_msg["raw_data"] = msg.raw_data;
    }
    else
    {
        daria_msg["data"] = msg.raw_data;
    }

    out << daria_msg.stringify();
}

}}
