#include "timeutils.h"

#include <time.h>

#include <string>
#include <sstream>
#include <iomanip>
#include <iostream>

namespace yxiva {

using std::setw;
using std::setfill;

const char* monthNames[] = { "января", "февраля", "марта",    "апреля",  "мая",    "июня",
                             "июля",   "августа", "сентября", "октября", "ноября", "декабря" };

const char* monthShortNames[] = { "янв", "фев", "мар", "апр", "май", "июн",
                                  "июл", "авг", "сен", "окт", "ноя", "дек" };

void extractDate(const char* dateInString, struct tm* dateTime)
{
    char skip;
    int fullYear;

    std::stringstream ss(dateInString);
    ss >> setw(2) >> dateTime->tm_mday >> skip >> setw(2) >> dateTime->tm_mon >> skip >> setw(4) >>
        fullYear >> std::ws;

    dateTime->tm_year = fullYear - 1900;
    dateTime->tm_mon -= 1;
    dateTime->tm_isdst = -1;

    ss >> setw(2) >> dateTime->tm_hour >> skip >> setw(2) >> dateTime->tm_min >> skip >> setw(2) >>
        dateTime->tm_sec;
}

void getNow(tm* now)
{
    now->tm_isdst = -1;
    const time_t nowUnixTime = time(NULL);
    gmtime_r(&nowUnixTime, now);
}

void shiftDate(tm* dateTime, int shiftInSeconds)
{
    const time_t shifted = mktime(dateTime) + shiftInSeconds;
    gmtime_r(&shifted, dateTime);
}

string toShortDate(const struct tm* dateTime, const struct tm* now)
{

    std::stringstream ss;
    ss << setfill('0');

    if ((dateTime->tm_mday == now->tm_mday) && (dateTime->tm_mon == now->tm_mon) &&
        (dateTime->tm_year == now->tm_year))
    {
        ss << setw(2) << dateTime->tm_hour << ":" << setw(2) << dateTime->tm_min;
    }
    else
    {
        if (dateTime->tm_year == now->tm_year)
        {
            ss << dateTime->tm_mday << " " << monthShortNames[dateTime->tm_mon] << ".";
        }
        else
        {
            ss << setw(2) << dateTime->tm_mday << "." << setw(2) << (dateTime->tm_mon + 1) << "."
               << setw(2) << ((dateTime->tm_year + 1900) % 100);
        }
    }

    return ss.str();
}

string toISODate(const struct tm* dateTime)
{
    std::stringstream ss;
    ss << setfill('0');

    ss << setw(4) << (dateTime->tm_year + 1900) << '-' << setw(2) << (dateTime->tm_mon + 1) << '-'
       << setw(2) << (dateTime->tm_mday) << 'T' << setw(2) << dateTime->tm_hour << ':' << setw(2)
       << dateTime->tm_min << ':' << setw(2) << dateTime->tm_sec;

    return ss.str();
}

string toFullDate(const struct tm* dateTime)
{
    std::stringstream ss;
    ss << setfill('0');

    ss << dateTime->tm_mday << ' ' << monthNames[dateTime->tm_mon] << ' '
       << (dateTime->tm_year + 1900) << " в " << setw(2) << dateTime->tm_hour << ':' << setw(2)
       << dateTime->tm_min;

    return ss.str();
}

string fromStrDateToShortDate(const char* dateInString, int shiftInSeconds)
{
    struct tm dateTime, now;

    extractDate(dateInString, &dateTime);
    shiftDate(&dateTime, shiftInSeconds);
    getNow(&now);

    return toShortDate(&dateTime, &now);
}

string fromStrDateToFullDate(const char* dateInString, int shiftInSeconds)
{
    struct tm dateTime;

    extractDate(dateInString, &dateTime);
    shiftDate(&dateTime, shiftInSeconds);

    return toFullDate(&dateTime);
}

string fromStrDateToISODate(const char* dateInString, int shiftInSeconds)
{
    struct tm dateTime;

    extractDate(dateInString, &dateTime);
    shiftDate(&dateTime, shiftInSeconds);

    return toISODate(&dateTime);
}

}
