#pragma once

#include "common.h"
#include <yplatform/yield.h>
#include <boost/format.hpp>

namespace yxiva::web::idm {

template <typename Stream>
struct add_role
{
    std::shared_ptr<struct settings> settings;
    Stream stream;
    string subject_type;
    string role;
    string login;
    boost::shared_ptr<service_manager> service_manager_ptr;
    boost::shared_ptr<yhttp::call> http_client_ptr;

    string project;
    string env;
    string role_type;
    tvm_app_info tvm_app;
    std::shared_ptr<struct service_data> service_data;
    boost::system::error_code err_code;
    yhttp::response resp;
    operation::result res;

    add_role(
        std::shared_ptr<struct settings> settings,
        Stream stream,
        const string& subject_type,
        const string& role,
        const string& login)
        : settings(settings), stream(stream), subject_type(subject_type), role(role), login(login)
    {
        service_manager_ptr = yplatform::find<service_manager>(settings->service_manager);
        http_client_ptr = yplatform::find<yhttp::call>(settings->http_client);
    }

    using yield_context = yplatform::yield_context<add_role<Stream>>;

    void operator()(yield_context yield_ctx)
    {
        reenter(yield_ctx)
        {
            std::tie(res, project, env, role_type) = parse_role(role);
            if (!res) return send_bad_request(stream, res);

            std::tie(res, tvm_app.id) = parse_login(login);
            if (!res) return send_bad_request(stream, res);

            res = validate_args(subject_type, env, role_type);
            if (!res) return send_bad_request(stream, res);

            yield request_tvm_info(tvm_app, yield_ctx.capture(err_code, resp));
            if (!err_code)
            {
                std::tie(res, tvm_app.name) = process_tvm_info_response(resp);
                if (!res)
                {
                    report_bad_tvm_info_response(stream, res.error_reason);
                    tvm_app.name = "unknown";
                }
            }
            else
            {
                report_tvm_info_error(stream, err_code);
                tvm_app.name = "unknown";
            }

            yield add_tvm_app(project, role_type, env, tvm_app, yield_ctx.capture(res));
            if (!res)
            {
                if (res.error_reason == "service doesn't exist")
                {
                    return send_bad_request(stream, "project not found");
                }
                return send_internal_error(stream, res);
            }

            send_ok_response(stream);
        }
    }

    void operator()(std::exception_ptr exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const std::exception& e)
        {
            send_internal_error(stream, operation::result{ e.what() });
        }
    }

    template <typename Handler>
    void request_tvm_info(const tvm_app_info& tvm_app, const Handler& handler)
    {
        auto tvm_info_url = (boost::format(settings->tvm_info_url) % tvm_app.id).str();
        http_client_ptr->async_run(
            stream->request()->context,
            yhttp::request::GET(tvm_info_url),
            settings->http_options,
            handler);
    }

    void report_tvm_info_error(Stream stream, const boost::system::error_code& err_code)
    {
        YLOG_CTX_GLOBAL(stream->request()->context, info)
            << "tvm app info request failed: error=" << err_code.message();
    }

    void report_bad_tvm_info_response(Stream stream, const string& info)
    {
        YLOG_CTX_GLOBAL(stream->request()->context, info)
            << "tvm app info request failed: error=" << info;
    }

    std::tuple<operation::result, string> process_tvm_info_response(const yhttp::response& resp)
    {
        if (resp.status != 200) return { "http status " + std::to_string(resp.status), "" };

        json_value json_resp;
        if (auto parse_err = json_resp.parse(resp.body)) return { *parse_err, "" };

        auto status = json_resp["status"].to_string();
        if (status != "ok") return { "status " + status, "" };

        auto resp_app_name = json_resp["name"].to_string();
        if (resp_app_name.empty()) return { "empty name", "" };

        return { "", resp_app_name };
    }

    template <typename Handler>
    void add_tvm_app(
        const string& service_name,
        const string& role_type,
        const string& env,
        const tvm_app_info& tvm_app,
        const Handler& handler)
    {
        auto add_app = [role_type, env, tvm_app](auto properties) {
            if (role_type == "publisher")
            {
                properties.tvm_publishers[env].insert(tvm_app);
            }
            else
            {
                properties.tvm_subscribers[env].insert(tvm_app);
            }
            return properties;
        };
        service_manager_ptr->update_service_properties(
            stream->request()->context, service_name, std::move(add_app), handler);
    }
};

}

#include <yplatform/unyield.h>
