#pragma once

#include "common.h"
#include <yplatform/yield.h>

namespace yxiva::web::idm {

template <typename Stream>
struct remove_role
{
    std::shared_ptr<struct settings> settings;
    Stream stream;
    string subject_type;
    string role;
    string login;
    boost::shared_ptr<service_manager> service_manager_ptr;

    string project;
    string env;
    string role_type;
    tvm_app_info tvm_app;
    std::shared_ptr<struct service_data> service_data;
    boost::system::error_code err_code;
    yhttp::response resp;
    operation::result res;

    remove_role(
        std::shared_ptr<struct settings> settings,
        Stream stream,
        const string& subject_type,
        const string& role,
        const string& login)
        : settings(settings), stream(stream), subject_type(subject_type), role(role), login(login)
    {
        service_manager_ptr = yplatform::find<service_manager>(settings->service_manager);
    }

    using yield_context = yplatform::yield_context<remove_role<Stream>>;

    void operator()(yield_context yield_ctx)
    {
        reenter(yield_ctx)
        {
            std::tie(res, project, env, role_type) = parse_role(role);
            if (!res) return send_bad_request(stream, res);

            std::tie(res, tvm_app.id) = parse_login(login);
            if (!res) return send_bad_request(stream, res);

            res = validate_args(subject_type, env, role_type);
            if (!res) return send_bad_request(stream, res);

            yield erase_service(project, role_type, env, tvm_app, yield_ctx.capture(res));
            if (!res)
            {
                if (res.error_reason == "service doesn't exist")
                {
                    return send_bad_request(stream, "project not found");
                }
                return send_internal_error(stream, res);
            }

            send_ok_response(stream);
        }
    }

    void operator()(std::exception_ptr exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const std::exception& e)
        {
            send_internal_error(stream, operation::result{ e.what() });
        }
    }

    template <typename Handler>
    void erase_service(
        const string& service_name,
        const string& role_type,
        const string& env,
        const tvm_app_info& tvm_app,
        const Handler& handler)
    {
        auto erase_app = [role_type, env, tvm_app](auto properties) {
            if (role_type == "publisher")
            {
                properties.tvm_publishers[env].erase(tvm_app);
            }
            else
            {
                properties.tvm_subscribers[env].erase(tvm_app);
            }
            return properties;
        };
        service_manager_ptr->update_service_properties(
            stream->request()->context, service_name, std::move(erase_app), handler);
    }
};

}

#include <yplatform/unyield.h>
