#include "impl.h"

#include "api/send.h"
#include "api/subscribe.h"
#include "api/subscribe_app.h"
#include "api/subscribe_url.h"
#include "api/subscriptions_user.h"
#include "api2/batch_send.h"
#include "api2/secret_sign.h"
#include "api2/send.h"
#include "api2/subscribe.h"
#include "api2/subscribe_app.h"
#include "api2/subscribe_url.h"
#include "api2/subscribe_webpush.h"
#include "api2/list.h"
#include "api2/unsubscribe_app.h"
#include "api2/unsubscribe_webpush.h"
#include "api2/vapid_key.h"
#include "api2/apns_queue_repeat.h"
#include "api2/unsubscribe.h"
#include "api2/watch_subscribers.h"
#include "api2/unsubscribe_websocket.h"
#include "auth/secure_sign.h"
#include "backapi/list.h"
#include "backapi/list_applications.h"
#include "backapi/notify.h"
#include "backapi/notify_all.h"
#include "backapi/overview.h"
#include "backapi/sign.h"
#include "backapi/ws_subscribe.h"
#include "backapi/check_token.h"
#include "backapi/reload_xconf.h"
#include "bind_arguments_handler.h"
#include "extapi/mobile_push.h"
#include "extapi/schedule_unsubscribe_app.h"
#include "extapi/bb.h"
#include "http_method_handler.h"
#include "idm/idm.h"
#include "idm/add_role.h"
#include "idm/remove_role.h"
#include "methods/aux_methods.h"
#include "methods/doc.h"
#include "methods/events.h"
#include "methods/ping.h"
#include "arguments/fetch_history.h"
#include "arguments/service_and_filter.h"
#include "arguments/uid.h"
#include "arguments/validator_with_size.h"
#include "secure_handler.h"
#include "web/auth/api.h"
#include "web/auth/cookie.h"
#include "webui/webui.h"
#include "auth/abc_authorization.h"
#include "webpushapi/internal_notify.h"
#include "webpushapi/send.h"
#include "webpushapi/subscribe.h"
#include "webpushapi/unsubscribe.h"
#include "webpushapi/list.h"
#include "webpushapi/monitor.h"
#include "webpushapi/list_catalogue.h"

#include "web/formatters/json.h"
#include "web/formatters/daria.h"
#include "utils/bindings.h"
#include <yxiva/core/split.h>
#include <ymod_webserver/server.h>
#include <ymod_webserver/methods/transform.h>
#include <yplatform/net/streambuf.h>
#include <yplatform/find.h>
#include <iterator>
#include <cctype>

namespace yxiva { namespace web {

namespace {

bool starts_with(const string& str, const string& prefix)
{
    return prefix.size() <= str.size() && std::equal(prefix.begin(), prefix.end(), str.begin());
}

struct strong_order_splitter
{
    std::vector<string> operator()(const string& value) const
    {
        std::vector<string> result = utils::split(value, ",");
        return result;
    }
};

struct uniq_splitter
{
    std::vector<string> operator()(const string& value) const
    {
        std::vector<string> result = utils::split_unique(value, ",");
        return result;
    }
};
}

impl::impl(const yplatform::ptree& ptree) : settings_(new settings)
{
    settings_->load(ptree);
}

void impl::init()
{
    formatters_kit_.add("daria", formatters::formatter_ptr(new formatters::daria));
    formatters_kit_.add("json", formatters::formatter_ptr(new formatters::json));

    stats_.reset(new request_stats);

    webpushapi_catalogue_ = std::make_shared<webpushapi::catalogue>();
    webpushapi_catalogue_->logger(logger());

    auto webserver = yplatform::find<ymod_webserver::server>(settings_->xiva_webserver);
    auto public_ep = settings_->xiva_endpoint;

    webserver->bind(public_ep, { "*load-limit*" }, methods::load_limit);
    webserver->bind_websocket(public_ep, { "*load-limit*" }, methods::load_limit_websocket);

    // use custom key extractor while webserver doesnt support paths with wildcards
    webserver->set_custom_key_extractor(
        public_ep, std::bind(&impl::custom_key_extractor, this, std::placeholders::_1));

    // pre-configure RPC
    auto rpc_typed_logger =
        yplatform::log::tskv_logger(YGLOBAL_LOG_SERVICE, settings_->websocket_rpc.log_id);
    websocketapi_rpc_handler_ = std::make_shared<websocket_rpc::main_handler>(
        settings_, stats_, logger(), rpc_typed_logger);
    webpushapi_rpc_handler_ = std::make_shared<websocket_rpc::main_handler>(
        settings_, stats_, logger(), rpc_typed_logger);

    bind_http_api();
    bind_back_api();
    bind_v1_api();
    bind_v2_api();
    bind_webpush_api();
    bind_webui_api();
    bind_idm_api();
}

void impl::bind_http_api()
{
    namespace p = std::placeholders;

    auto webserver = yplatform::find<ymod_webserver::server>(settings_->xiva_webserver);
    auto public_ep = settings_->xiva_endpoint;

    webserver->bind(public_ep, { "/ping" }, methods::ping);
    webserver->bind(
        public_ep,
        { "/events/sse" },
        std::bind(methods::events_sse, p::_1, settings_, formatters_kit_, true));
    webserver->bind(
        public_ep,
        { "/" },
        methods::doc(
            "/", settings_->api.doc_root, settings_->api.root_regex, settings_->api.root_default));
    webserver->bind(
        public_ep,
        { "/doc", "/doc/*" },
        methods::doc(
            "/doc", settings_->api.doc_root, settings_->api.doc_regex, settings_->api.doc_default));
    webserver->bind(public_ep, { "/doc*" }, [](auto stream) {
        stream->set_code(ymod_webserver::codes::moved_permanently);
        stream->add_header("Location", "https://console.push.yandex-team.ru");
        stream->result_body({});
    });
    webserver->bind_websocket(
        public_ep,
        { "/events/websocket" },
        std::bind(methods::events_websocket, settings_, formatters_kit_, p::_1, true));
}

void impl::bind_back_api()
{
    auto webserver = yplatform::find<ymod_webserver::server>(settings_->back_api_webserver);
    auto private_ep = settings_->back_api_endpoint;

    webserver->bind(private_ep, { "/notify/*" }, bind_arguments(backapi::notify, settings_));
    webserver->bind(
        private_ep,
        { "/webpushapi/notify" },
        bind_arguments(webpushapi::internal_notify, settings_, webpushapi_catalogue_));
    webserver->bind(
        private_ep,
        { "/webpushapi/list_catalogue" },
        bind_arguments(webpushapi::list_catalogue, webpushapi_catalogue_));
    webserver->bind(private_ep, { "/list" }, backapi::list);
    webserver->bind(private_ep, { "/notify_all" }, backapi::notify_all);
    webserver->bind(private_ep, { "/ping" }, methods::ping);
    webserver->bind(private_ep, { "/sign" }, backapi::sign);
    webserver->bind(private_ep, { "/overview" }, backapi::overview);
    webserver->bind(
        private_ep, { "/ws_subscribe" }, bind_arguments(backapi::ws_subscribe, settings_));
    webserver->bind(
        private_ep,
        { "/list_applications" },
        bind_arguments(backapi::list_applications, settings_));
    webserver->bind(
        private_ep, { "/check_token" }, bind_arguments(backapi::check_token, settings_));
    webserver->bind(
        private_ep, { "/reload_xconf" }, bind_arguments(backapi::reload_xconf, settings_));
    webserver->set_custom_key_extractor(private_ep, [](http_stream_ptr stream) {
        auto full_path = stream->request()->url.make_full_path();
        return starts_with(full_path, "/notify/") ? "/notify/*" : full_path;
    });
}

void impl::bind_v1_api()
{
    namespace p = std::placeholders;
    using ymod_webserver::argument;
    using ymod_webserver::default_validator;
    using ymod_webserver::transformer;
    using auth::xtoken;
    using auth::xtoken_check_type;

    auto auth_service_manager = find_service_manager(settings_->api.auth_service_manager);
    auto webserver = yplatform::find<ymod_webserver::server>(settings_->xiva_webserver);
    auto public_ep = settings_->xiva_endpoint;

    auto stoken = xtoken<xtoken_check_type::send>{ auth_service_manager, { "token", "stoken" } };
    auto ltoken = xtoken<xtoken_check_type::listen>{ auth_service_manager, { "token", "ctoken" } };
    auto auth_stoken = auth_factory<service_authorization, false>(settings_, stoken);
    auto auth_ltoken = auth_factory<service_authorization, true>(settings_, ltoken);
    auto service_with_filter_argument = argument<service_with_filter>(
        "service", default_validator<string>(), service_with_filter_converter());
    auto service_argument = argument<string>("service");
    webserver->bind(public_ep, { "/v1/send" }, POST_handler(auth_stoken(api::send)));
    webserver->bind(
        public_ep,
        { "/v1/subscribe/url" },
        POST_handler(auth_ltoken(api::subscribe_url)),
        transformer(service_with_filter_argument));
    webserver->bind(
        public_ep,
        { "/v1/subscribe/app" },
        POST_handler(std::bind(api::subscribe_app, p::_1, settings_)));
    webserver->bind(
        public_ep,
        { "/v1/subscribe/sse" },
        GET_handler(std::bind(api::subscribe_sse, p::_1, settings_)));
    webserver->bind(
        public_ep, { "/v1/subscriptions/user" }, GET_handler(auth_stoken(api::subscriptions_user)));

    // API v1: extentions
    webserver->bind(
        public_ep,
        { "/ext/v1/mobile_push/apns" },
        POST_handler(auth_stoken(extapi::mobile_push_apns)));
    webserver->bind(
        public_ep,
        { "/ext/v1/mobile_push/gcm", "/ext/v1/mobile_push/fcm" }, // gcm_compatibility
        POST_handler(auth_stoken(extapi::mobile_push_fcm)));
    webserver->bind(
        public_ep,
        { "/ext/v1/mobile_push/mpns" },
        POST_handler(auth_stoken(extapi::mobile_push_mpns)));
    webserver->bind(
        public_ep,
        { "/ext/v1/mobile_push/wns" },
        POST_handler(auth_stoken(extapi::mobile_push_wns)));
    webserver->bind(
        public_ep,
        { "/ext/v1/mobile_batch_push/gcm", "/ext/v1/mobile_batch_push/fcm" }, // gcm_compatibility
        POST_handler(auth_stoken(extapi::mobile_batch_push_fcm)));

    webserver->bind_websocket(
        public_ep,
        { "/v1/subscribe", "/v1/subscribe/websocket" },
        std::bind(api::subscribe_websocket, p::_1, settings_));
}

void impl::bind_v2_api()
{
    namespace p = std::placeholders;
    using ymod_webserver::argument;
    using ymod_webserver::default_validator;
    using ymod_webserver::optional_argument;
    using ymod_webserver::validator;
    using ymod_webserver::transformer;
    using auth::tvm;
    using auth::tvm_check_type;
    using auth::xtoken;
    using auth::xtoken_check_type;

    auto auth_service_manager = find_service_manager(settings_->api.auth_service_manager);
    auto auth_tvm =
        yplatform::find<ymod_tvm::tvm2_module, std::shared_ptr>(settings_->api.auth_tvm);
    auto bbclient = yplatform::find<ymod_blackbox::client>("bbclient");
    auto webserver = yplatform::find<ymod_webserver::server>(settings_->xiva_webserver);
    auto public_ep = settings_->xiva_endpoint;

    auto stoken = xtoken<xtoken_check_type::send>{ auth_service_manager };
    auto stoken_stream = xtoken<xtoken_check_type::send_stream>{ auth_service_manager };
    auto ltoken = xtoken<xtoken_check_type::listen>{ auth_service_manager };
    auto ltoken_no_service = xtoken<xtoken_check_type::listen_no_service>{ auth_service_manager };
    auto oauth = auth::oauth{ bbclient };
    auto multi_xtoken = auth::multi_xtoken{ { auth_service_manager } };
    auto tvm_publisher = tvm<tvm_check_type::publisher>{ auth_tvm, auth_service_manager };
    auto tvm_stream_publisher =
        tvm<tvm_check_type::stream_publisher>{ auth_tvm, auth_service_manager };
    auto tvm_subscriber = tvm<tvm_check_type::subscriber>{ auth_tvm, auth_service_manager };
    auto tvm_multi_subscriber =
        tvm<tvm_check_type::multi_subscriber>{ auth_tvm, auth_service_manager };
    auto auth_stvm_stoken =
        auth_factory<service_authorization, true>(settings_, tvm_publisher, stoken);
    auto auth_ltvm_ltoken =
        auth_factory<service_authorization, true>(settings_, tvm_subscriber, ltoken);
    auto auth_ltvm_ltoken_no_service =
        auth_factory<service_authorization, true>(settings_, tvm_subscriber, ltoken_no_service);
    auto auth_stream_stvm_stoken =
        auth_factory<service_authorization, true>(settings_, tvm_stream_publisher, stoken_stream);
    auto auth_ltvm_ltoken_oauth =
        auth_factory<varying_authorization, true>(settings_, tvm_subscriber, ltoken, oauth);
    auto auth_lstvm_lstoken = auth_factory<service_authorization, true>(
        settings_, any_from(tvm_publisher, tvm_subscriber), any_from(stoken, ltoken));
    auto auth_multi_ltvm_ltoken = auth_factory<multi_service_authorization, true>(
        settings_, tvm_multi_subscriber, multi_xtoken);

    auto service_list_argument =
        argument<std::vector<string>>("service", validator(isalnum_ext_list), uniq_splitter());
    auto service_with_filter_argument = argument<service_with_filter>(
        "service", default_validator<string>(), service_with_filter_converter());
    auto service_with_filter_list_argument = argument<service_with_filter_list>(
        "service", default_validator<string>(), service_with_filter_list_converter());

    auto optional_uid =
        optional_argument<string>("user (uid)", { "uid", "user" }, "", uid_validator());
    auto required_uid = argument<string>("user (uid)", { "uid", "user" }, uid_validator());
    auto required_uid_list = argument<std::vector<string>>(
        "user (uid)", { "uid", "user" }, uid_list_validator(), uniq_splitter());
    auto optional_topic = optional_argument<string>("topic", "", uid_validator());
    auto optional_topic_list =
        optional_argument<std::vector<string>>("topic", {}, uid_list_validator(), uniq_splitter());
    auto required_topic_list =
        argument<std::vector<string>>("topic", uid_list_validator(), uniq_splitter());
    auto required_push_token = argument<string>(
        "push_token", validator_with_size(isprint, settings_->api.max_push_token_length));
    auto required_device_id = argument<string>(
        "device id",
        { "device", "device_id" },
        validator_with_size(isalnum_ext, settings_->api.max_device_id_length));
    auto optional_device_id = optional_argument<string>(
        "device id",
        { "device", "device_id" },
        "",
        validator_with_size(isalnum_ext, settings_->api.max_device_id_length));
    auto required_service = argument<string>("service", validator(isalnum_ext));
    auto optional_service = optional_argument<string>("service", "", validator(isalnum_ext));
    auto required_client = argument<string>("client", validator(isalnum_ext));
    auto optional_client = optional_argument<string>("client", "", validator(isalnum_ext));
    auto required_session = argument<string>("session", validator(isalnum_ext));
    auto optional_session = optional_argument<string>("session", "", validator(isalnum_ext));
    auto required_session_or_uuid =
        argument<string>("uuid", { "uuid", "session" }, validator(isalnum_ext));
    auto optional_extra = optional_argument<string>(
        "extra", string{}, validator_with_size(isprint, settings_->api.max_extra_length));
    auto optional_ext_id = optional_argument<string>(
        "external request id",
        { "external_request_id", "request_id" },
        "",
        validator_with_size(isalnum_ext, settings_->api.max_external_id_length));
    auto subscribe_transformer = transformer(
        required_uid_list,
        optional_topic,
        service_with_filter_list_argument,
        required_client,
        required_session,
        optional_argument<string>("sign", ""),
        optional_argument<time_t>("ts", 0),
        fetch_history_argument("fetch_history"),
        fetch_history_argument("try_fetch_history"),
        optional_argument<string>("filter", ""));
    auto watch_subscribers_transformer = transformer(
        required_service,
        required_topic_list,
        optional_uid,
        optional_argument<string>("sign", ""),
        optional_argument<time_t>("ts", 0));

    auto isalnum_event = [](char c) { return isalnum_ext(c) || c == ' '; };
    webserver->bind(
        public_ep,
        { "/v2/send" },
        POST_handler(auth_stvm_stoken(api2::single_send)),
        transformer(
            optional_service,
            optional_uid,
            optional_topic,
            argument<string>("event", validator(isalnum_event)),
            optional_argument<uint32_t>("ttl", 604800),
            optional_argument<string>("session", "", validator(isalnum_ext)),
            optional_ext_id));

    webserver->bind(
        public_ep,
        { "/v2/batch_send" },
        POST_handler(auth_stvm_stoken(api2::batch_send)),
        transformer(
            optional_service,
            argument<string>("event", validator(isalnum_ext)),
            optional_argument<uint32_t>("ttl", 604800 /*1 week*/),
            optional_ext_id));

    webserver->bind(
        public_ep,
        { "/v2/stream_send" },
        POST_handler(auth_stream_stvm_stoken(api2::stream_send)),
        transformer(
            optional_service,
            argument<string>("event", validator(isalnum_ext)),
            optional_argument<uint32_t>("ttl", 604800)));

    webserver->bind(
        public_ep,
        { "/v2/subscribe/app" },
        POST_handler(auth_ltvm_ltoken_oauth(pass(coroutine<api2::subscribe_app>(), settings_))),
        transformer(
            service_with_filter_argument,
            optional_uid,
            optional_argument<string>("filter", ""),
            argument<string>("app_name", validator(isalnum_ext)),
            required_push_token,
            required_session_or_uuid,
            argument<string>("platform", validator(isalnum_ext)),
            optional_extra,
            optional_client,
            optional_device_id,
            optional_topic));
    webserver->bind(
        public_ep,
        { "/v2/subscribe/url" },
        POST_handler(auth_ltvm_ltoken(api2::subscribe_url)),
        transformer(
            service_with_filter_argument,
            argument<string>("callback"),
            required_uid,
            optional_argument<string>("filter", ""),
            optional_client,
            required_session,
            optional_topic));
    webserver->bind(
        public_ep,
        { "/v2/unsubscribe/app" },
        POST_handler(auth_ltvm_ltoken_oauth(api2::unsubscribe_app(settings_))),
        transformer(required_service, optional_uid, required_session_or_uuid, optional_topic));
    webserver->bind(
        public_ep,
        { "/v2/subscriptions/user", "/v2/subscriptions", "/v2/list" },
        GET_handler(auth_lstvm_lstoken(api2::list)),
        transformer(
            optional_argument<string>("service", "", validator(isalnum_ext)),
            optional_uid,
            optional_topic));

    webserver->bind(
        public_ep,
        { "/v2/unsubscribe", "/v2/unsubscribe/url" },
        POST_handler(auth_ltvm_ltoken(api2::unsubscribe)),
        transformer(
            required_service, optional_uid, optional_topic, argument<string>("subscription_id")));

    webserver->bind(
        public_ep,
        { "/v2/secret_sign" },
        GET_handler(auth_multi_ltvm_ltoken(api2::sign)),
        transformer(
            service_list_argument,
            required_uid_list,
            optional_topic_list,
            optional_argument<std::time_t>("ts", 0)));

    webserver->bind(
        public_ep,
        { "/v2/verify_secret_sign" },
        GET_handler(auth_multi_ltvm_ltoken(api2::verify_sign)),
        transformer(
            service_list_argument,
            required_uid_list,
            optional_topic_list,
            argument<string>("sign"),
            argument<std::time_t>("ts")));

    webserver->bind(
        public_ep, { "/v2/subscribe/sse" }, api2::subscribe_sse(settings_), subscribe_transformer);

    webserver->bind_websocket(
        public_ep,
        { "/v2/subscribe/websocket" },
        api2::subscribe_websocket(settings_),
        subscribe_transformer);

    webserver->bind(
        public_ep,
        { "/beta/wild/send" },
        POST_handler(auth_stvm_stoken(api2::wild_send)),
        transformer(
            optional_service,
            argument<string>("event", validator(isalnum_ext)),
            optional_argument<uint32_t>("ttl", 604800),
            optional_session));

    webserver->bind(public_ep, { "/v2/vapid_key" }, api2::vapid_key(settings_));

    webserver->bind(
        public_ep,
        { "/v2/subscribe/webpush" },
        POST_handler(auth_ltvm_ltoken(pass(coroutine<api2::subscribe_webpush>(), settings_))),
        transformer(
            service_with_filter_argument,
            required_uid,
            required_client,
            required_session_or_uuid,
            argument<string>(
                "subscription",
                [](const string& sub) {
                    json_value json_out;
                    auto error = json_out.parse(sub);
                    return !error;
                }),
            optional_argument<string>("filter", ""),
            optional_extra,
            optional_topic,
            optional_argument<unsigned>("ttl", settings_->api.hub.subscribe_webpush_ttl)));

    webserver->bind(
        public_ep,
        { "/v2/unsubscribe/webpush" },
        POST_handler(auth_ltvm_ltoken(api2::unsubscribe_webpush(settings_))),
        transformer(required_service, optional_uid, required_session_or_uuid, optional_topic));

    webserver->bind(
        public_ep,
        { "/ext/v2/schedule_unsubscribe/app" },
        POST_handler(extapi::schedule_unsubscribe_app),
        transformer(
            required_service,
            argument<string>("platform", validator(isalnum_ext)),
            required_session_or_uuid));

    webserver->bind(
        public_ep,
        { "/ext/v2/bb/login" },
        POST_handler(auth_ltvm_ltoken_no_service(extapi::bb_login)),
        transformer(
            optional_service,
            argument<string>("user", validator(isalnum_ext)),
            argument<string>("yandexuid", validator(isalnum_ext)),
            optional_argument<string>("bb_connection_id", "", validator(isalnum_ext))));

    webserver->bind(
        public_ep,
        { "/ext/v2/bb/logout" },
        POST_handler(auth_ltvm_ltoken_no_service(extapi::bb_logout)),
        transformer(
            optional_service,
            argument<string>("user", validator(isalnum_ext)),
            argument<string>("yandexuid", validator(isalnum_ext))));

    webserver->bind(
        public_ep,
        { "/v2/apns_queue_repeat" },
        auth_ltvm_ltoken_no_service(api2::apns_queue_repeat(settings_)),
        transformer(
            optional_service,
            argument<string>("app_name", validator(isalnum_ext)),
            required_device_id,
            argument<local_id_t>(
                "last notification position (pos)", { "last_seen", "pos", "position" }),
            argument<local_id_t>("count")));

    webserver->bind_websocket(
        public_ep,
        { "/v2/watch/subscribers" },
        api2::watch_subscribers(settings_),
        watch_subscribers_transformer);

    webserver->bind_websocket(
        public_ep,
        { "/websocketapi" },
        std::bind(&websocket_rpc::main_handler::handle, websocketapi_rpc_handler_, p::_1));

    websocketapi_rpc_handler_->bind({ "/ping" }, [](std::shared_ptr<websocket_rpc::stream> stream) {
        stream->result(ymod_webserver::codes::ok, "pong");
    });

    websocketapi_rpc_handler_->bind(
        { "/subscribe" },
        api2::subscribe_websocketapi(settings_),
        transformer(
            required_uid,
            optional_topic,
            service_with_filter_argument,
            required_client,
            required_session,
            optional_argument<string>("sign", ""),
            optional_argument<time_t>("ts", 0),
            fetch_history_argument("fetch_history"),
            fetch_history_argument("try_fetch_history"),
            optional_argument<string>("filter", "")));

    websocketapi_rpc_handler_->bind(
        { "/unsubscribe" },
        api2::unsubscribe_websocketapi(settings_),
        transformer(
            optional_uid,
            optional_topic,
            required_service,
            argument<string>("subscription_token"),
            optional_argument<string>("sign", ""),
            optional_argument<time_t>("ts", 0)));
}

void impl::bind_webpush_api()
{
    namespace p = std::placeholders;
    using ymod_webserver::argument;
    using ymod_webserver::optional_argument;
    using ymod_webserver::transformer;
    using ymod_webserver::validator;

    auto webserver = yplatform::find<ymod_webserver::server>(settings_->xiva_webserver);
    auto public_ep = settings_->xiva_endpoint;

    webpushapi_rpc_handler_->bind({ "/ping" }, [](std::shared_ptr<websocket_rpc::stream> stream) {
        stream->result(ymod_webserver::codes::ok, "pong");
    });

    if (settings_->webpushapi.enabled)
    {
        auto required_uuid = argument<string>("uuid", validator(isalnum_ext));

        webserver->bind(
            public_ep,
            { "/webpushapi/send/*" },
            POST_handler(bind_arguments(webpushapi::send, settings_)));
        webpushapi_rpc_handler_->bind(
            { "/webpushapi/subscribe" },
            bind_arguments_websocket(webpushapi::subscribe, settings_),
            transformer(
                optional_argument<string>("subscription_set", ""),
                argument<string>("public_key"),
                required_uuid));
        webpushapi_rpc_handler_->bind(
            { "/webpushapi/unsubscribe" },
            bind_arguments_websocket(webpushapi::unsubscribe, settings_),
            transformer(argument<string>("subscription")));
        webpushapi_rpc_handler_->bind(
            { "/webpushapi/list" },
            bind_arguments_websocket(webpushapi::list, settings_),
            transformer(argument<string>("subscription_set")));
        webpushapi_rpc_handler_->bind(
            { "/webpushapi/monitor" },
            bind_arguments_websocket(webpushapi::monitor, settings_, webpushapi_catalogue_),
            transformer(argument<string>("subscription_set")));
    }

    webserver->bind_websocket(
        public_ep,
        { "/webpushapi/json_rpc" },
        std::bind(&websocket_rpc::main_handler::handle, webpushapi_rpc_handler_, p::_1));
}

void impl::bind_webui_api()
{
    using ymod_webserver::argument;
    using ymod_webserver::optional_argument;
    using ymod_webserver::validator;
    using ymod_webserver::transformer;

    auto webserver = yplatform::find<ymod_webserver::server>(settings_->xiva_webserver);
    if (settings_->api.webui.enabled)
    {
        auto webui_ep = settings_->xiva_endpoint;
        auto unauthorized = webui::unauthorized(settings_);
        auto environment = argument<string>("environment", { "env" }, validator(isalnum));
        auto optional_owner = optional_argument<string>(
            "owner", { "uid", "user", "owner" }, "", validator(isalnum_ext));
        auto optional_owner_type =
            optional_argument<string>("owner_type", "", validator(isalnum_ext));

        webserver->bind(
            webui_ep,
            { "/webui/list" },
            auth_with_cookie(settings_, abc_auth(webui::list()), unauthorized));
        webserver->bind(
            webui_ep,
            { "/webui/service/create" },
            POST_handler(auth_with_cookie(
                settings_,
                abc_auth(webui::webui_auth_admin_only(webui::service_create())),
                unauthorized)),
            transformer(optional_owner_type, optional_owner));
        webserver->bind(
            webui_ep,
            { "/webui/service/update" },
            POST_handler(auth_with_cookie(
                settings_, abc_auth(webui::webui_auth(webui::service_update())), unauthorized)),
            transformer(optional_owner_type, optional_owner));
        webserver->bind(
            webui_ep,
            { "/webui/service/revoke" },
            POST_handler(auth_with_cookie(
                settings_, abc_auth(webui::webui_auth(webui::service_update(true))), unauthorized)),
            transformer(optional_owner_type, optional_owner));
        webserver->bind(
            webui_ep,
            { "/webui/send_token/update" },
            POST_handler(auth_with_cookie(
                settings_, abc_auth(webui::webui_auth(webui::send_token_update())), unauthorized)),
            transformer(optional_owner_type, optional_owner, environment));
        webserver->bind(
            webui_ep,
            { "/webui/send_token/revoke" },
            POST_handler(auth_with_cookie(
                settings_,
                abc_auth(webui::webui_auth(webui::send_token_update(true))),
                unauthorized)),
            transformer(optional_owner_type, optional_owner, environment));
        webserver->bind(
            webui_ep,
            { "/webui/listen_token/update" },
            POST_handler(auth_with_cookie(
                settings_,
                abc_auth(webui::webui_auth(webui::listen_token_update())),
                unauthorized)),
            transformer(optional_owner_type, optional_owner, environment));
        webserver->bind(
            webui_ep,
            { "/webui/listen_token/revoke" },
            POST_handler(auth_with_cookie(
                settings_,
                abc_auth(webui::webui_auth(webui::listen_token_update(true))),
                unauthorized)),
            transformer(optional_owner_type, optional_owner, environment));
        webserver->bind(
            webui_ep,
            { "/webui/app/info" },
            GET_handler(auth_with_cookie(
                settings_,
                abc_auth(webui::webui_auth_admin_only(webui::app_info())),
                unauthorized)),
            transformer(
                optional_owner_type,
                optional_owner,
                argument<string>("app", validator(isalnum_ext)),
                argument<string>("platform", validator(isalnum_ext))));
        webserver->bind(
            webui_ep,
            { "/webui/app/update" },
            POST_handler(auth_with_cookie(
                settings_, abc_auth(webui::webui_auth(webui::app_update())), unauthorized)),
            transformer(optional_owner_type, optional_owner));
        webserver->bind(
            webui_ep,
            { "/webui/app/revert" },
            POST_handler(auth_with_cookie(
                settings_, abc_auth(webui::webui_auth(webui::app_revert())), unauthorized)),
            transformer(optional_owner_type, optional_owner));
        webserver->bind(
            webui_ep,
            { "/webui/app/revoke" },
            POST_handler(auth_with_cookie(
                settings_, abc_auth(webui::webui_auth(webui::app_update(true))), unauthorized)),
            transformer(optional_owner_type, optional_owner));
    }
}

void impl::bind_idm_api()
{
    using ymod_webserver::optional_argument;
    using ymod_webserver::transformer;

    auto webserver = yplatform::find<ymod_webserver::server>(settings_->xiva_webserver);
    auto endpoint = settings_->xiva_endpoint;
    auto idm_conf = std::make_shared<idm::settings>(settings_->api.idm);
    if (idm_conf->enabled)
    {
        auto subject_type_argument = optional_argument<string>("subject_type", "");
        auto role_argument = optional_argument<string>("role", "");
        auto login_argument = optional_argument<string>("login", "");

        webserver->bind(endpoint, { "/idm/info" }, check_tvm(idm_conf->tvm, idm::info(idm_conf)));
        webserver->bind(
            endpoint,
            { "/idm/get-all-roles" },
            check_tvm(idm_conf->tvm, idm::get_all_roles(idm_conf)));
        webserver->bind(
            endpoint,
            { "/idm/add-role" },
            check_tvm(idm_conf->tvm, pass(coroutine<idm::add_role<http_stream_ptr>>(), idm_conf)),
            transformer(subject_type_argument, role_argument, login_argument));
        webserver->bind(
            endpoint,
            { "/idm/remove-role" },
            check_tvm(
                idm_conf->tvm, pass(coroutine<idm::remove_role<http_stream_ptr>>(), idm_conf)),
            transformer(subject_type_argument, role_argument, login_argument));
    }
}

void impl::start()
{
}

void impl::reload(const yplatform::ptree& ptree)
{
    settings_->reload(ptree);
}

void impl::stop()
{
    websocketapi_rpc_handler_->shutdown();
    webpushapi_rpc_handler_->shutdown();
}

void impl::fini(void)
{
}

string impl::make_sign(const string& data, time_t expired) const
{
    return make_secure_sign(data, expired, settings_->sign_secret);
}

string impl::custom_key_extractor(ymod_webserver::http::stream_ptr stream)
{
    static const std::vector<string> prefixes = { "/doc", "/webpushapi/send/" };

    auto current_rps = stats_->new_request();
    if (stream->request()->url.path.size() && stream->request()->url.path[0] != "ping" &&
        current_rps > static_cast<int>(settings_->rps_limit))
    {
        return "*load-limit*";
    }
    stats_->accept_request();

    auto result = stream->request()->url.make_full_path();

    for (const auto& prefix : prefixes)
    {
        if (starts_with(result, prefix))
        {
            result = prefix + "*";
            break;
        }
    }
    return result;
}
}}

#include <yplatform/module_registration.h>
REGISTER_MODULE(yxiva::web::impl)
