#include "doc.h"

#include <yxiva/core/read_text_file.h>
#include <boost/filesystem.hpp>
#include <boost/regex.hpp>
#include <algorithm>
#include <numeric>

namespace yxiva { namespace web { namespace methods {
namespace {
boost::filesystem::path make_relative_path(
    const boost::filesystem::path& path,
    const boost::filesystem::path& root)
{
    return std::accumulate(
        std::mismatch(path.begin(), path.end(), root.begin()).first,
        path.end(),
        boost::filesystem::path{},
        std::divides<boost::filesystem::path>());
}
}

doc::doc(
    const string& location,
    const string& root,
    const string& regex,
    const string& default_file)
{
    if (location != "/") location_ = location;

    if (!boost::filesystem::exists(root)) return;

    boost::regex filter(regex);
    using boost::filesystem::recursive_directory_iterator;
    using boost::filesystem::is_regular_file;
    boost::filesystem::path root_path(root);
    recursive_directory_iterator end;
    for (recursive_directory_iterator it(root); it != end; ++it)
    {
        string filename = it->path().filename().string();
        string extension = it->path().extension().string();
        boost::smatch what;
        if (is_regular_file(it->status()) && boost::regex_match(filename, what, filter))
        {
            string mime_type = "text/html";
            if (extension == ".css")
            {
                mime_type = "text/css";
            }
            else if (extension == ".js")
            {
                mime_type = "application/javascript";
            }
            else if (extension == ".png")
            {
                mime_type = "image/png";
            }
            else if (extension == ".jpg")
            {
                mime_type = "image/jpeg";
            }
            auto relative_filename = make_relative_path(it->path(), root_path).string();
            resources_[location_ + "/" + relative_filename] = {
                mime_type, read_text_file(root + "/" + relative_filename)
            };
            // Put default record with location conforming to make_full_path behaviour.
            if (default_file.size() && relative_filename == default_file)
            {
                resources_[location_.empty() ? "/" : location_] = {
                    mime_type, read_text_file(root + "/" + relative_filename)
                };
            }
        }
    }
}

void doc::operator()(ymod_webserver::http::stream_ptr stream)
{
    auto path = stream->request()->url.make_full_path();
    auto p_resource = resources_.find(path);
    if (p_resource != resources_.end())
    {
        stream->set_code(http_codes::ok);
        stream->set_content_type(p_resource->second.mime);
        stream->result_body(p_resource->second.data);
    }
    else
    {
        stream->result(http_codes::not_found, "");
    }
}

}}}
