#include "events.h"

#include "web/auth/user.h"
#include "web/auth/secure_sign.h"

#include "web/formatters/kit.h"
#include "web/formatters/daria.h"
#include "web/formatters/json.h"
#include <yxiva/core/print_date.h>

#include "web/sse_subscriber.h"
#include "web/control_session.h"
#include "web/websocket_subscriber.h"
#include "web/util.h"
#include "subscribe_process.h"

namespace yxiva { namespace web { namespace methods {

namespace {

bool check_proto_version(request_ptr req, std::pair<int, int> version)
{
    if (req->proto_version.first < version.first || req->proto_version.second < version.second)
    {
        return false;
    }
    return true;
}

}

void on_auth_finished(
    user_info_future_t auth_result,
    request_ptr req,
    settings_ptr /*settings*/,
    formatters::kit formatters_kit,
    subscribe_process_ptr process_object) noexcept
{
    req->context->profilers.pop("passport");
    if (auth_result.has_exception())
    {
        process_object->send_auth_failed(get_exception_reason(auth_result));
        return;
    }
    user_info ui = auth_result.get();

    string format = (req->url.param_value("format", "daria"));
    if (!formatters_kit.has(format))
    {
        process_object->send_bad_request("unsupported format", get_exception_reason(auth_result));
        return;
    }

    channels_set decoded_requests;
    try
    {
        decoded_requests = decode(req, ui);
    }
    catch (std::exception const& e)
    {
        process_object->send_bad_request("bad service argument", e.what());
        return;
    }

    if (decoded_requests.empty())
    {
        process_object->send_bad_request("no such channel(s)", "");
        return;
    }

    formatters::formatter_ptr formatter = formatters_kit.get(format);
    time_t timestamp = req->url.param_value_cast<time_t>("ts", 0);

    web_subscriber_ptr subs = process_object->create_subscriber(
        req->context, formatter, timestamp, extract_client_id(req));

    timer_ptr timer = process_object->make_timer();
    control_session(
        ui,
        decoded_requests,
        subs,
        timer,
        process_object->ping_interval(),
        process_object->future_session_close());
}

bool valid_sign(request_ptr req, settings_ptr settings)
{
    string id = req->url.param_value("uid", "");
    string sign = req->url.param_value("sign", "");
    time_t timestamp = req->url.param_value_cast<time_t>("ts", 0);

    return timestamp != 0 && std::time(nullptr) < timestamp &&
        sign == make_secure_sign(uid_data(id), timestamp, settings->sign_secret);
}

void events(
    request_ptr req,
    settings_ptr settings,
    formatters::kit formatters_kit,
    subscribe_process_ptr process_object,
    bool check_sign)
{
    if (check_sign && !valid_sign(req, settings))
    {
        process_object->send_sign_error();
        return;
    }

    req->context->profilers.push("passport");

    user_info_future_t auth_res = auth_user(req);

    auth_res.add_callback(
        std::bind(&on_auth_finished, auth_res, req, settings, formatters_kit, process_object));
}

void events_sse(
    ymod_webserver::response_ptr resp,
    settings_ptr settings,
    formatters::kit formatters_kit,
    bool check_sign)
{
    ymod_webserver::request_ptr req = resp->request();
    // SSE requires HTTP v1.1
    if (!check_proto_version(req, std::make_pair(1, 1)))
    {
        send_http_error(
            req->context,
            resp,
            ymod_webserver::codes::bad_request,
            "Protoсol version 1.1 required");
        return;
    }

    // start to track session close
    promise_close prom;
    resp->add_error_handler(boost::bind(&promise_close::set, prom));

    subscriber_factory_t factory = boost::bind(create_sse_subscriber, _1, resp, _2, _3, _4);
    time_duration ping_interval = settings->ss_event_ping_interval;
    // pass as a future object future_close(prom);
    subscribe_process_ptr process_object = ::yxiva::make_shared<subscribe_process_http>(
        req, resp, ping_interval, future_close(prom), factory);

    events(req, settings, formatters_kit, process_object, check_sign);
}

void events_websocket(
    settings_ptr settings,
    formatters::kit formatters_kit,
    ymod_webserver::websocket::output_stream_ptr output_stream,
    bool check_sign)
{
    promise_close promise_session_closed;
    output_stream->set_close_callback(boost::bind(&promise_close::set, promise_session_closed));
    future_close future_session_closed = promise_session_closed;

    ymod_webserver::request_ptr req = output_stream->request();
    string full_path = make_full_path(req->url.path);

    subscriber_factory_t factory = boost::bind(
        create_websocket_subscriber,
        _1,
        output_stream,
        _2,
        _3,
        settings->rand_websocket_inactive_timeout(),
        _4);

    subscribe_process_ptr process_object = ::yxiva::make_shared<subscribe_process_websocket>(
        req, output_stream, settings->websocket_ping_interval, future_session_closed, factory);
    events(req, settings, formatters_kit, process_object, check_sign);
}
}}}
