#pragma once

#include "web/find_deps.h"
#include "web/types.h"
#include <ymod_httpclient/call.h>

namespace yxiva { namespace web {

// Log hub error codes and hide them from clients.
template <typename Stream>
inline void handle_default_hub_codes(
    Stream&& stream,
    const boost::system::error_code& ec,
    yhttp::response response)
{
    if (stream->ctx()->is_cancelled()) return;
    if (ec || response.status == 500 || response.status == 502)
    {
        YLOG_CTX_GLOBAL(stream->ctx(), info) << "proxy call failed: error=" << ec.message();
        stream->result(http_codes::internal_server_error);
    }
    else if (response.status == http_codes::ok)
    {
        stream->result(http_codes::ok);
    }
    else if (response.status == http_codes::service_unavailable)
    {
        YLOG_CTX_GLOBAL(stream->ctx(), info) << "rate limit exceeded";
        stream->result(http_codes::too_many_requests, "request rate exceeded");
    }
    else
    {
        YLOG_CTX_GLOBAL(stream->ctx(), error)
            << "proxy call failed: status=" << response.status << " error=" << response.body;
        stream->result(http_codes::internal_server_error);
    }
}

template <typename Stream>
inline auto handle_default_hub_codes(Stream&& stream)
{
    return [stream](const boost::system::error_code& ec, yhttp::response response) {
        handle_default_hub_codes(stream, ec, std::move(response));
    };
}

template <typename Stream, typename Handler>
struct hub_list_json_handler
{
    void operator()(const boost::system::error_code& ec, yhttp::response response)
    {
        if (ec || response.status != 200) return handle_default_hub_codes(stream_, ec, response);

        json_value list;
        if (auto error = list.parse(response.body, json_type::tarray))
        {
            return send_internal_error(stream_, "", "failed to parse hub list: " + *error);
        }
        std::vector<sub_t> subs;
        subs.reserve(list.size());
        for (auto&& item : list.array_items())
        {
            subs.resize(subs.size() + 1);
            subs.back().id = item["id"].to_string();
            subs.back().uid = item["uid"].to_string();
            subs.back().session_key = item["session_key"].to_string();
            subs.back().ttl = item["ttl"].get<ttl_t>(0);
        }
        // TODO try-catch
        handler_(std::move(subs));
    }

    Stream stream_;
    typename std::decay<Handler>::type handler_;
};

template <typename Stream, typename Handler>
hub_list_json_handler<Stream, Handler> handle_hub_list_json(const Stream& stream, Handler&& handler)
{
    return { stream, std::forward<Handler>(handler) };
}

template <typename Stream, typename Handler>
struct hub_list_size_handler
{
    void operator()(const boost::system::error_code& ec, yhttp::response response)
    {
        if (ec || response.status != 200) return handle_default_hub_codes(stream_, ec, response);

        json_value list;
        if (auto error = list.parse(response.body, json_type::tarray))
        {
            return send_internal_error(stream_, "", "failed to parse hub list: " + *error);
        }
        // TODO try-catch
        handler_(list.size());
    }

    Stream stream_;
    typename std::decay<Handler>::type handler_;
};

template <typename Stream, typename Handler>
hub_list_size_handler<Stream, Handler> handle_hub_list_size(const Stream& stream, Handler&& handler)
{
    return { stream, std::forward<Handler>(handler) };
}

}}