#include "subscribe_process.h"

#include "web/util.h"

namespace yxiva { namespace web { namespace methods {

void send_ws_error(
    task_context_ptr context,
    ymod_webserver::websocket::output_stream_ptr stream,
    uint16_t code,
    string const& message = "",
    string const& verbose_message = "")
{
    std::ostringstream ss;
    ss << code;
    if (message.size())
    {
        ss << " " << message;
    }
    if (verbose_message.size())
    {
        ss << " " << verbose_message;
    }
    YLOG_CTX_GLOBAL(context, info) << ss.str();
    stream->close_connection(code, message);
}

void send_ws_json_error(
    task_context_ptr context,
    ymod_webserver::websocket::output_stream_ptr stream,
    uint16_t code,
    string const& message = "",
    string const& verbose_message = "")
{
    send_ws_error(context, stream, code, "{ \"Error\": \"" + message + "\" }\n", verbose_message);
}

subscribe_process::subscribe_process(
    ymod_webserver::request_ptr req,
    time_duration ping_interval,
    future_close future_session_closed,
    subscriber_factory_t subscriber_factory)
    : req_(req)
    , ping_interval_(ping_interval)
    , future_session_closed_(future_session_closed)
    , subscriber_factory_(subscriber_factory)
{
}

subscribe_process_http::subscribe_process_http(
    ymod_webserver::request_ptr req,
    ymod_webserver::response_ptr response,
    time_duration ping_interval,
    future_close future_session_closed,
    subscriber_factory_t subscriber_factory)
    : subscribe_process(req, ping_interval, future_session_closed, subscriber_factory)
    , response_(response)
{
}

void subscribe_process_http::send_retry_after(std::time_t time)
{
    response_->add_header("Retry-After", std::time(0) + time);
    send_http_error(req_->context, response_, ymod_webserver::codes::service_unavailable, "", "");
}

void subscribe_process_http::send_sign_error()
{
    send_http_error(req_->context, response_, ymod_webserver::codes::forbidden, "Bad sign");
}

void subscribe_process_http::send_auth_failed(const string& info)
{
    send_http_error(
        req_->context, response_, ymod_webserver::codes::unauthorized, "Auth failed", info);
}

void subscribe_process_http::send_bad_request(const string& info, const string& priv_info)
{
    send_http_error(req_->context, response_, ymod_webserver::codes::unauthorized, info, priv_info);
}

timer_ptr subscribe_process_http::make_timer()
{
    return response_->make_timer();
}

subscribe_process_websocket::subscribe_process_websocket(
    ymod_webserver::request_ptr req,
    ymod_webserver::websocket::output_stream_ptr output_stream,
    time_duration ping_interval,
    future_close future_session_closed,
    subscriber_factory_t subscriber_factory)
    : subscribe_process(req, ping_interval, future_session_closed, subscriber_factory)
    , output_stream_(output_stream)
{
}

void subscribe_process_websocket::send_retry_after(std::time_t time)
{
    send_ws_error(
        req_->context,
        output_stream_,
        ymod_webserver::websocket::codes::retry_after,
        "Retry-After: " + boost::lexical_cast<string>(time),
        "");
}

void subscribe_process_websocket::send_sign_error()
{
    send_ws_json_error(
        req_->context, output_stream_, ymod_webserver::websocket::codes::forbidden, "bad sign");
}

void subscribe_process_websocket::send_auth_failed(const string& info)
{
    send_ws_json_error(
        req_->context,
        output_stream_,
        ymod_webserver::websocket::codes::unauthorized,
        "authorization failed",
        info);
}

void subscribe_process_websocket::send_bad_request(const string& info, const string& priv_info)
{
    send_ws_json_error(
        req_->context,
        output_stream_,
        ymod_webserver::websocket::codes::bad_request,
        info,
        priv_info);
}

timer_ptr subscribe_process_websocket::make_timer()
{
    return output_stream_->make_timer();
}

}}}
