#pragma once

#include "types.h"
#include "rproxy/rproxy.h"
#include "mod_log/find.h"

namespace yxiva::web {

template <typename... Args>
inline void enable_rproxy(http_stream_ptr, Args&&...)
{
}

template <typename... Args>
inline void enable_rproxy(websocket_rpc::stream_ptr, Args&&...)
{
}

template <typename Properties>
inline void enable_rproxy(
    websocket_stream_ptr stream,
    settings_ptr settings,
    const string& uid,
    Properties properties)
{
    namespace bin = binary_protocol;
    using status_frame = bin::proxy_status_frame;
    using errc = bin::error_code;

    auto rproxy = find_rproxy();
    if (!rproxy->enabled_for_any(properties->services)) return;

    auto running_requests_count = std::make_shared<size_t>(0);
    boost::weak_ptr<ymod_webserver::websocket::stream> weak_stream(stream);
    stream->add_message_callback([ctx = stream->ctx(),
                                  weak_stream,
                                  uid,
                                  properties,
                                  rproxy,
                                  running_requests_count,
                                  settings](const ymod_webserver::websocket::message& msg) {
        auto stream = weak_stream.lock();
        if (!stream) return;

        // Ignore text messages.
        if (msg.opcode != ymod_webserver::websocket::message::opcode_binary) return;

        if (*running_requests_count >= settings->rproxy_concurrency_limit_per_session)
        {
            find_xivaws_log()->log_rproxy_request(
                ctx,
                to_string(errc::too_many_requests),
                "",
                0,
                "too many requests in session",
                msg.length);
            stream->send_binary(bin::pack(status_frame{ 0, errc::too_many_requests }));
            return;
        }
        ++(*running_requests_count);

        string user_url;
        auto&& in_headers = stream->request()->headers;
        if (auto it = in_headers.find("x-rproxy-url"); it != in_headers.end())
        {
            user_url = it->second;
        }

        string data;
        data.reserve(msg.length);
        data.assign(msg.data.begin(), msg.data.end());

        string out_headers = "X-Ya-User: " + uid + "\r\n" +
            "X-RProxy-Connection: " + ctx->uniq_id() +
            "\r\n"
            "X-RProxy-Session: " +
            properties->session +
            "\r\n"
            "X-RProxy-Client: " +
            properties->client +
            "\r\n"
            "X-Forwarded-For: " +
            ctx->remote_address +
            "\r\n"
            "X-Real-IP: " +
            ctx->remote_address +
            "\r\n"
            "X-Real-Port: " +
            std::to_string(ctx->remote_port) + "\r\n";

        rproxy->async_call(
            ctx,
            properties->strong_order_services,
            user_url,
            out_headers,
            std::move(data),
            stream->get_io_service().wrap([weak_stream, running_requests_count](string data) {
                --(*running_requests_count);
                if (auto stream = weak_stream.lock())
                {
                    stream->send_binary(std::move(data));
                }
            }));
    });
}

}
