#include "web/settings.h"
#include "web/find_deps.h"
#include <yxiva/core/split.h>
#include <yxiva/core/callbacks.h>

namespace yxiva { namespace web {

void settings::load(const yplatform::ptree& ptree)
{
    sign_secret = ptree.get("sign_secret_new", ptree.get("sign_secret", ""));
    if (sign_secret.empty())
    {
        throw std::domain_error("missing or empty sign secret phrase");
    }
    ss_event_ping_interval = time_traits::seconds(ptree.get("ss_event_ping_interval", 15));
    websocket_ping_interval = time_traits::seconds(ptree.get("websocket_ping_interval", 300));
    websocket_inactive_timeout = ptree.get<time_duration>("websocket_inactive_timeout");
    websocket_inactive_timeout_deviation =
        ptree.get<time_duration>("websocket_inactive_timeout_deviation");
    xiva_webserver = ptree.get("xiva_webserver", "");
    xiva_endpoint = ptree.get("xiva_endpoint", "");
    back_api_webserver = ptree.get("back_api_webserver", "");
    back_api_endpoint = ptree.get("back_api_endpoint", "");
    retry_after_response_value = ptree.get<std::time_t>("retry_response_value", 3600);
    rps_limit = ptree.get<unsigned>("rps_limit", 30000);
    disable_authentication = ptree.get<bool>("disable_authentication", false);

    service_features.load(ptree.get_child("service_features"));

    api.strict_token_check_mode = ptree.get<bool>("api.tokens.strict_token_check_on");

    api.message_max_tags = ptree.get<unsigned>("api.message_max_tags", 10);
    api.message_max_payload = ptree.get<unsigned>("api.message_max_payload", 64 * 1024 * 1024);
    api.max_filter_length = ptree.get<unsigned>("api.max_filter_length");
    api.max_push_token_length = ptree.get<unsigned>("api.max_push_token_length", 400);
    api.max_extra_length = ptree.get<unsigned>("api.max_extra_length", 256);
    api.max_external_id_length = ptree.get<unsigned>("api.max_external_id_length", 80);
    api.max_xiva_token_length = ptree.get<unsigned>("api.max_xiva_token_length", 100);
    api.max_device_id_length = ptree.get<unsigned>("api.max_device_id_length", 256);

    webpushapi.load(ptree.get_child("webpushapi"));

    utils::split(api.user_oauth_scopes, ptree.get("api.user_oauth_scopes", ""), ", ");
    utils::split(api.register_oauth_scopes, ptree.get("api.register_oauth_scopes", ""), ", ");
    api.sign_ttl = ptree.get<std::time_t>("api.sign_ttl");
    api.vapid_key = read_text_file(ptree.get<string>("api.vapid_public"));

    api.hub.load(ptree.get_child("api.hub"));

    api.doc_root = ptree.get<string>("api.doc_root");
    api.doc_regex = ptree.get<string>("api.doc_regex");
    api.doc_default = ptree.get<string>("api.doc_default");
    api.root_regex = ptree.get<string>("api.root_regex");
    api.root_default = ptree.get<string>("api.root_default");

    api.back.load(ptree.get_child("api.back"));

    api.idm.load(ptree.get_child("api.idm"));

    load_origin_domains(ptree.get("origin_domains", ""), origin_domains);

    websocket_rpc.load(ptree.get_child("websocket_rpc"));

    api.auth_service_manager = ptree.get<string>("api.auth_service_manager");
    // Check auth service manager exists.
    if (auto service_manager = find_service_manager(api.auth_service_manager))
    {
        if (service_manager->environment().empty())
        {
            throw std::runtime_error("auth service manager must have defined environment");
        }
    }
    else
    {
        throw std::runtime_error(
            "auth service manager module not found: " + api.auth_service_manager);
    }
    api.auth_tvm = ptree.get<string>("api.auth_tvm");
    // Check auth tvm module exists (throws if module not found).
    yplatform::find_module<ymod_tvm::tvm2_module>(api.auth_tvm);

    api.webui.load(ptree.get_child_optional("api.webui"));

    api.apns_queue.load(ptree.get_child_optional("api.apns_queue"));

    api.watch_subscribers.load(ptree.get_child("api.watch_subscribers"));

    api.abc.load(ptree.get_child_optional("api.abc"));

    rproxy_concurrency_limit_per_session.store(
        ptree.get<size_t>("rproxy_concurrency_limit_per_session"));
}

void settings::reload(const yplatform::ptree& ptree)
{
    rproxy_concurrency_limit_per_session.store(
        ptree.get<size_t>("rproxy_concurrency_limit_per_session"));
}

void settings::api::hub::load(const yplatform::ptree& ptree)
{
    subscribe_app_ttl = ptree.get<unsigned>("subscribe_app_ttl");
    subscribe_webpush_ttl = ptree.get<unsigned>("subscribe_webpush_ttl");
    subscribe_wns_ttl = ptree.get<unsigned>("subscribe_wns_ttl");
    subscribe_bb_login_ttl = ptree.get<unsigned>("subscribe_bb_login_ttl");

    batch_send_timeout = ptree.get<time_duration>("batch_send_timeout");
    batch_size = ptree.get<size_t>("batch_size");
    batch_min_size = ptree.get<size_t>("batch_min_size");
    batch_max_recipients = ptree.get<size_t>("batch_max_recipients");
    max_message_ttl = ptree.get<uint32_t>("max_message_ttl");

    batch_buckets.load(ptree.get_child("batch_buckets"));
}

void settings::api::hub::batch_buckets::load(const yplatform::ptree& ptree)
{
    small = ptree.get<size_t>("small");
    medium = ptree.get<size_t>("medium");
}

void settings::webpushapi::load(const yplatform::ptree& ptree)
{
    enabled = ptree.get<bool>("enabled", enabled);
    if (enabled)
    {
        service = ptree.get<string>("service");
        max_subset_size = ptree.get<size_t>("max_subset_size");
        endpoint_characters_shown =
            ptree.get<size_t>("endpoint_characters_shown", endpoint_characters_shown);
        subscription_id = ptree.get<string>("subscription_id");
        subscription_ttl = ptree.get<ttl_t>("subscription_ttl");
        message_ttl = ptree.get<ttl_t>("message_ttl");
        auto base_url = ptree.get<string>("scheme", "https") + "://" +
            boost::asio::ip::host_name() + ptree.get<string>("suffix");
        callback = callback_uri::xiva_websocket_uri(base_url);
        push_resource_url = ptree.get<string>("push_resource_url");
        message_url = ptree.get<string>("message_url");
        if (push_resource_url.empty())
        {
            // It doesn't work for local configurations
            // because of custom ports they are listening.
            push_resource_url = "https://" + boost::asio::ip::host_name() + "/webpushapi/send/";
        }
        else if (push_resource_url.back() != '/')
        {
            push_resource_url += '/';
        }
        limits.public_key = ptree.get("limits.public_key", limits.public_key);
        limits.subset = ptree.get("limits.subset", limits.subset);
        limits.uidset_uuid = ptree.get("limits.uidset_uuid", limits.uidset_uuid);
    }
}

void settings::api::abc::load(const boost::optional<const yplatform::ptree&>& ptree)
{
    if (ptree)
    {
        enabled = true;
        auto oauth_token_file = ptree->get("oauth_token_file", "");
        if (oauth_token_file.size())
        {
            oauth_token = read_text_file(oauth_token_file);
        }
        auto roles_range = ptree->equal_range("roles");
        for (auto& it = roles_range.first; it != roles_range.second; ++it)
        {
            if (roles.size()) roles += ',';
            roles += it->second.data();
        }
    }
    else
    {
        enabled = false;
    }
}

void settings::api::webui::load(const boost::optional<const yplatform::ptree&>& ptree)
{
    if (ptree)
    {
        enabled = true;
        auth_redirect = ptree->get<string>("auth_redirect");
        auth_prefix = ptree->get<string>("auth_prefix");
        service_create_prefix = ptree->get<string>("service_create_prefix");
        max_name_length = ptree->get<unsigned>("max_name_length", max_name_length);
        max_description_length =
            ptree->get<unsigned>("max_description_length", max_description_length);
        max_scope_count = ptree->get<unsigned>("max_scope_count", max_scope_count);
        auto admin_range = ptree->equal_range("admin_uids");
        for (auto& it = admin_range.first; it != admin_range.second; ++it)
        {
            admins.insert(it->second.data());
        }
        service_manager = ptree->get<string>("service_manager");
    }
    else
    {
        enabled = false;
    }
}

void settings::api::apns_queue::load(const boost::optional<const yplatform::ptree&>& ptree)
{
    enabled = ptree && ptree->get<bool>("queue_enabled", enabled);
    if (enabled)
    {
        service = ptree->get<string>("service");
        max_repeat_count = ptree->get<size_t>("max_repeat_count");
    }
}

void settings::api::watch_subscribers::load(const yplatform::ptree& ptree)
{
    list_interval = ptree.get<time_duration>("list_interval");
    max_age = ptree.get<time_duration>("max_age");
    auto enabled_for_range = ptree.equal_range("enabled_for");
    for (auto& it = enabled_for_range.first; it != enabled_for_range.second; ++it)
    {
        enabled_for.insert(it->second.data());
    }
    max_topics = ptree.get<unsigned>("max_topics");
}

void settings::websocket_rpc::load(const yplatform::ptree& ptree)
{
    log_id = ptree.get<string>("log_id");
    tls_only = ptree.get("tls_only", tls_only);
    session_max_messages = ptree.get("session_max_messages", session_max_messages);
}

}}
