#pragma once

#include "load_text_file.h"
#include "origin_domains.h"
#include "web/auth/tokens.h"
#include "web/idm/settings.h"
#include <service_manager/interface.h>
#include <yxiva/core/types.h>
#include <yxiva/core/feature_toggling.h>
#include <yxiva/core/read_text_file.h>
#include <yplatform/util/split.h>
#include <map>
#include <set>
#include <vector>

namespace yxiva { namespace web {

struct back_api_settings
{
    void load(const yplatform::ptree& conf)
    {
        ws_subscribe_html = load_text_file(conf.get("ws_subscribe_html", ""));
    }

    yplatform::net::buffers::const_chunk_buffer ws_subscribe_html;
};

struct service_features
{
    blacklisted_feature api2_respond_subscription_id;
    whitelisted_feature session_as_ws_subscription_id;
    whitelisted_feature message_unlimited_payload;
    whitelisted_feature high_priority_websockets;

    void load(const yplatform::ptree& conf)
    {
        api2_respond_subscription_id.load(conf, "api2_respond_subscription_id_blacklist");
        session_as_ws_subscription_id.load(conf, "session_as_ws_subscription_id_whitelist");
        message_unlimited_payload.load(conf, "message_unlimited_payload");
        high_priority_websockets.load(conf, "high_priority_websockets");
    }
};

struct settings
{
    string sign_secret;
    time_duration ss_event_ping_interval;
    time_duration websocket_ping_interval;
    time_duration websocket_inactive_timeout;
    time_duration websocket_inactive_timeout_deviation;
    string xiva_webserver;
    string xiva_endpoint;
    string back_api_webserver;
    string back_api_endpoint;
    origin_domains_list origin_domains;

    std::time_t retry_after_response_value;

    unsigned rps_limit;

    bool disable_authentication;

    struct service_features service_features;

    struct api
    {
        struct hub
        {
            void load(const yplatform::ptree& ptree);

            unsigned subscribe_app_ttl;
            unsigned subscribe_webpush_ttl;
            unsigned subscribe_wns_ttl;
            unsigned subscribe_bb_login_ttl;

            time_duration batch_send_timeout;
            size_t batch_size;
            size_t batch_min_size;
            size_t batch_max_recipients;
            uint32_t max_message_ttl;

            struct batch_buckets
            {
                void load(const yplatform::ptree& ptree);

                size_t small;
                size_t medium;
            } batch_buckets;
        } hub;

        unsigned message_max_tags;
        unsigned message_max_payload;
        unsigned max_filter_length;
        unsigned max_push_token_length;
        unsigned max_extra_length; // limit for parameter 'extra'
        unsigned max_external_id_length;
        unsigned max_xiva_token_length;
        unsigned max_device_id_length;
        bool strict_token_check_mode;
        std::set<string> user_oauth_scopes;
        std::set<string> register_oauth_scopes;
        std::time_t sign_ttl;
        string vapid_key;

        string doc_root;
        string doc_regex;
        string doc_default;
        string root_regex;
        string root_default;

        string auth_service_manager;
        string auth_tvm;

        struct abc
        {
            void load(const boost::optional<const yplatform::ptree&>& ptree);

            bool enabled;
            string roles;
            string oauth_token;
        } abc;

        struct webui
        {
            void load(const boost::optional<const yplatform::ptree&>& ptree);

            string auth_redirect;
            string auth_prefix;
            string service_create_prefix;
            std::set<string> admins;
            bool enabled;
            unsigned max_name_length = 50;
            unsigned max_description_length = 140;
            unsigned max_scope_count = 50;
            string service_manager;
        } webui;

        struct apns_queue
        {
            void load(const boost::optional<const yplatform::ptree&>& ptree);

            string service;
            bool enabled = false;
            size_t max_repeat_count;
        } apns_queue;

        struct watch_subscribers
        {
            void load(const yplatform::ptree& ptree);

            time_duration list_interval;
            time_duration max_age;
            std::set<string> enabled_for; // service names
            unsigned max_topics = 0;
        } watch_subscribers;

        back_api_settings back;
        idm::settings idm;
    } api;

    struct websocket_rpc
    {
        void load(const yplatform::ptree& ptree);

        string log_id;
        bool tls_only = true;
        unsigned session_max_messages = 100000;
    } websocket_rpc;

    struct webpushapi
    {
        void load(const yplatform::ptree& ptree);

        bool enabled = false;
        string service;
        size_t max_subset_size;
        size_t endpoint_characters_shown = 10;
        string subscription_id;
        ttl_t subscription_ttl;
        ttl_t message_ttl;
        string callback;
        string push_resource_url;
        string message_url;
        struct
        {
            size_t public_key = 512;
            size_t subset = 512;
            size_t uidset_uuid = 512;
        } limits;
    } webpushapi;

    std::atomic_size_t rproxy_concurrency_limit_per_session = 100;

    void load(const yplatform::ptree& ptree);
    void reload(const yplatform::ptree& ptree);

    time_duration rand_websocket_inactive_timeout()
    {
        auto deviation_msec = yplatform::time_traits::duration_cast<milliseconds>(
                                  websocket_inactive_timeout_deviation)
                                  .count();
        auto rand_static = static_cast<unsigned>(std::time(nullptr));
        deviation_msec = deviation_msec * rand_r(&rand_static) / RAND_MAX;
        return websocket_inactive_timeout + milliseconds(deviation_msec);
    }
};

typedef std::shared_ptr<settings> settings_ptr;

}}
