#pragma once

#include "types.h"
#include "future_close.h"
#include <yxiva/core/message.h>
#include <ymod_webserver/websocket.h>
#include <ymod_webserver/methods/default_answers.h>

namespace yxiva { namespace web {

using headers_pack = std::vector<std::pair<string, string>>;

template <typename Stream>
inline void send_bad_request(
    const Stream& stream,
    const string& info,
    const string& internal_info = string(),
    headers_pack&& headers = {})
{
    YLOG_CTX_GLOBAL(stream->ctx(), info)
        << "bad request: " << info << ", internal info: " << internal_info;
    stream->set_code(http_codes::bad_request);
    for (auto&& h : headers)
    {
        stream->add_header(h.first, h.second);
    }
    stream->result_body(info);
}

template <>
inline void send_bad_request<websocket_stream_ptr>(
    const websocket_stream_ptr& stream,
    const string& info,
    const string& internal_info,
    headers_pack&&)
{
    YLOG_CTX_GLOBAL(stream->ctx(), info)
        << "bad request: " << info << ", internal info: " << internal_info;
    stream->close_connection(websocket_codes::bad_request, info);
}

inline void send_gone(
    const http_stream_ptr& stream,
    const string& info,
    headers_pack&& headers = {})
{
    YLOG_CTX_GLOBAL(stream->ctx(), info) << "gone: " << info;
    stream->set_code(http_codes::gone);
    for (auto&& h : headers)
    {
        stream->add_header(h.first, h.second);
    }
    stream->result_body(info);
}

inline void send_gone(const websocket_stream_ptr& stream, const string& info, headers_pack&& = {})
{
    YLOG_CTX_GLOBAL(stream->ctx(), info) << "gone: " << info;
    stream->close_connection(websocket_codes::gone, info);
}

inline void send_unauthorized(
    const http_stream_ptr& stream,
    const string& info,
    headers_pack&& headers = {})
{
    YLOG_CTX_GLOBAL(stream->ctx(), info) << "unauthorized: " << info;
    stream->set_code(http_codes::unauthorized);
    for (auto&& h : headers)
    {
        stream->add_header(h.first, h.second);
    }
    stream->result_body(info);
}

inline void send_unauthorized(
    const websocket_stream_ptr& stream,
    const string& info,
    headers_pack&& = {})
{
    YLOG_CTX_GLOBAL(stream->ctx(), info) << "unauthorized: " << info;
    stream->close_connection(websocket_codes::unauthorized, info);
}

template <typename Stream>
inline void send_internal_error(
    const Stream& stream,
    const string& info,
    const string& internal_info = string(),
    headers_pack&& headers = {})
{
    YLOG_CTX_GLOBAL(stream->ctx(), info)
        << "internal error: " << info << ", internal_info: " << internal_info;
    stream->set_code(http_codes::internal_server_error);
    for (auto&& h : headers)
    {
        stream->add_header(h.first, h.second);
    }
    stream->result_body(info);
}

template <>
inline void send_internal_error<websocket_stream_ptr>(
    const websocket_stream_ptr& stream,
    const string& info,
    const string& internal_info,
    headers_pack&&)
{
    YLOG_CTX_GLOBAL(stream->ctx(), info)
        << "internal error: " << info << ", internal_info: " << internal_info;
    stream->close_connection(websocket_codes::internal_server_error, info);
}

inline void send_forbidden(
    const http_stream_ptr& stream,
    const string& info,
    headers_pack&& headers = {})
{
    YLOG_CTX_GLOBAL(stream->ctx(), info) << "forbidden: " << info;
    stream->set_code(http_codes::forbidden);
    for (auto&& h : headers)
    {
        stream->add_header(h.first, h.second);
    }
    stream->result_body(info);
}

inline void send_forbidden(
    const websocket_stream_ptr& stream,
    const string& info,
    headers_pack&& = {})
{
    YLOG_CTX_GLOBAL(stream->ctx(), info) << "forbidden: " << info;
    json_value message;
    message["error"] = info;
    stream->close_connection(websocket_codes::forbidden, message.stringify());
}

inline future_close make_future_close(const http_stream_ptr& stream)
{
    promise_close promise_session_closed;
    stream->add_error_handler(boost::bind(&promise_close::set, promise_session_closed));
    return promise_session_closed;
}

inline future_close make_future_close(const websocket_stream_ptr& stream)
{
    promise_close promise_session_closed;
    stream->set_close_callback(boost::bind(&promise_close::set, promise_session_closed));
    return promise_session_closed;
}

}}
