#pragma once

#include "types.h"

#include <yxiva/core/types.h>
#include <yplatform/encoding/url_encode.h>
#include <yplatform/util/sstream.h>
#include <boost/algorithm/string/join.hpp>

namespace yxiva { namespace web {

using sstream = yplatform::sstream;

inline void send_http_error(
    task_context_ptr context,
    ymod_webserver::response_ptr resp,
    ymod_webserver::codes::code code,
    string const& message = "",
    string const& verbose_message = "")
{
    std::ostringstream ss;
    ss << code;
    if (message.size())
    {
        ss << " " << message;
    }
    if (verbose_message.size())
    {
        ss << " " << verbose_message;
    }
    YLOG_CTX_GLOBAL(context, info) << ss.str();
    resp->set_code(code);
    resp->result_body(message);
}

inline string make_full_path(const ymod_webserver::uri_path_t& path)
{
    typedef ymod_webserver::uri_path_t::const_iterator iterator;

    string result = "/";
    iterator it = path.begin();
    while (it != path.end())
    {
        result.append(*it);
        ++it;
        if (it != path.end()) result.append("/");
    }
    return result;
}

inline string url_encode(const string& data)
{
    string encoded;
    encoded += yplatform::url_encode(data);
    return encoded;
}

inline void replace(string& source, const string& what, const string& text)
{
    auto pos = source.find(what);
    if (pos == string::npos) return;
    source.replace(pos, what.size(), text);
}

template <typename StreamPtr>
inline void protect_request_data(const StreamPtr& stream, const string& data)
{
    static const string SECRET = "secret";

    if (data.empty()) return;

    replace(stream->request()->raw_url, data, SECRET);
    replace(stream->request()->raw_request_line, data, SECRET);

    auto& params = stream->request()->url.params;
    auto it =
        std::find_if(params.begin(), params.end(), [&data](const std::pair<string, string>& param) {
            return param.second == data;
        });
    if (it != params.end())
    {
        it->second = SECRET;
    }
}

template <typename StreamPtr>
inline void protect_request_data(const StreamPtr& stream, const std::vector<string>& multi_data)
{
    for (auto&& data : multi_data)
    {
        protect_request_data(stream, data);
    }
}

template <typename StreamPtr, typename Range>
inline void attach_custom_data(const StreamPtr& stream, const string& key, const Range& data)
{
    stream->ctx()->custom_log_data[key] = boost::algorithm::join(data, ",");
}

inline string format_http_error(const boost::system::error_code& ec, int status)
{
    return ec ? ec.message() : "http code " + std::to_string(status);
}

inline void append_to_comma_separated_list(string& list, const string& str)
{
    if (list.size())
    {
        list += ", ";
    }
    list += str;
}

template <typename Strings>
string to_comma_separated_list(const Strings& strings)
{
    string csv;
    for (auto&& s : strings)
    {
        append_to_comma_separated_list(csv, s);
    }
    return csv;
}

}}