#pragma once

#include <yxiva/core/types.h>
#include <yxiva/core/gid.h>
#include <yxiva/core/methods/check.h>
#include <yplatform/task_context.h>
#include <yplatform/encoding/base64.h>
#include <yplatform/util/uuid_generator.h>
#include <boost/regex.hpp>
#include <boost/thread/tss.hpp>

namespace yxiva { namespace web { namespace webpushapi {

namespace {

// thread_local is broken in clang, __thread supports trivialy constructed types only.
static boost::thread_specific_ptr<yplatform::util::string_uuid_generator> uuid_generator;

inline string generate_uuid()
{
    if (!uuid_generator.get())
    {
        uuid_generator.reset(new yplatform::util::string_uuid_generator);
    }
    return (*uuid_generator)();
}

inline bool validate_uid(const string& uid)
{
    return validate_param(uid, [](char c) { return isalnum_ext(c) || c == UID_TAIL_DELIMITER; });
};
}

inline string generate_uidsetid(task_context_ptr ctx)
{
    // TODO also use external uuid?
    return ctx->uniq_id() + "." + generate_uuid();
}

inline string generate_uid(const string& uidsetid)
{
    return uidsetid + UID_TAIL_DELIMITER + generate_uuid();
}

inline bool decode_uidset_from_uid(const string& uid, string& uidsetid)
{
    auto pos = uid.find(UID_TAIL_DELIMITER);
    if (pos == string::npos) return false;
    uidsetid.assign(uid.begin(), uid.begin() + pos);
    return true;
}

inline string encode_subset(const string& uidsetid)
{
    // TODO encrypt id
    string res;
    res += yplatform::base64_urlsafe_encode(uidsetid.begin(), uidsetid.end());
    std::reverse(res.begin(), res.end());
    return res;
}

inline string decode_uidsetid_from_subset(string subset)
{
    // TODO encrypt id
    string res;
    std::reverse(subset.begin(), subset.end());
    res += yplatform::base64_urlsafe_decode(subset.begin(), subset.end());
    return validate_uid(res) ? res : string();
}

inline string encode_subscription(const string& uid)
{
    // TODO encrypt id
    string res;
    res += yplatform::base64_urlsafe_encode(uid.begin(), uid.end());
    std::reverse(res.begin(), res.end());
    return res;
}

inline operation::result decode_subscription(
    const string& subscription,
    string& uidsetid,
    string& uid)
{
    // TODO encrypt id
    string res = subscription;
    std::reverse(res.begin(), res.end());
    uid.clear();
    uid += yplatform::base64_urlsafe_decode(res.begin(), res.end());
    if (!validate_uid(uid)) return "bad characters in uid";
    return decode_uidset_from_uid(uid, uidsetid) ? operation::success : "invalid uid";
}

// Expects url with tailing '/'.
inline string encode_push_resource(const string& uid, const string& url)
{
    // TODO encrypt id
    string xored = uid;
    for (auto& c : xored)
        c = c ^ 0x8;
    string res;
    res += yplatform::base64_urlsafe_encode(xored.begin(), xored.end());
    std::reverse(res.begin(), res.end());
    return url + res;
}

inline operation::result decode_push_resource(
    const string& push_resource,
    string& uidsetid,
    string& uid)
{
    // TODO encrypt id
    try
    {
        string xored_base64 = push_resource;
        std::reverse(xored_base64.begin(), xored_base64.end());
        uid.clear();
        uid += yplatform::base64_urlsafe_decode(xored_base64.begin(), xored_base64.end());
        for (auto& c : uid)
            c = c ^ 0x8;

        if (!validate_uid(uid)) return "bad characters in uid";
        if (!decode_uidset_from_uid(uid, uidsetid))
        {
            return "cant decode uidset";
        }
    }
    catch (const std::exception& ex)
    {
        return ex.what();
    }
    return operation::success;
}

}}}