#pragma once

#include "catalogue.h"
#include "common.h"
#include "encoding.h"

namespace yxiva { namespace web { namespace webpushapi {

void internal_notify(
    settings_ptr /*settings*/,
    std::shared_ptr<catalogue> catalogue,
    const http_stream_ptr& stream)
{
    auto&& log_data = stream->ctx()->custom_log_data;
    log_data["api_group"] = "webpushapi notifications";

    auto&& req = stream->request();
    auto&& raw_body = req->raw_body;
    string source(raw_body.begin(), raw_body.end());
    message msg;
    try
    {
        unpack(source, msg);
    }
    catch (const std::exception& e)
    {
        send_bad_webpushapi_request(stream, "notify", "can't decode message", e.what());
        return;
    }

    log_data["transit_id"] = msg.transit_id;
    if (msg.uid.empty())
    {
        send_bad_webpushapi_request(stream, "notify", "no uid specified");
        return;
    }

    string uidset;
    if (!decode_uidset_from_uid(msg.uid, uidset))
    {
        send_bad_webpushapi_request(stream, "notify", "invalid uid");
        return;
    }

    auto client_stream = catalogue->find(uidset);
    if (!client_stream)
    {
        log_data["api_result"] = "notify deactivated";
        stream->result(http_codes::no_content, "");
        return;
    }

    string payload;
    payload += yplatform::base64_urlsafe_encode(msg.raw_data.begin(), msg.raw_data.end());

    json_value result;
    result["subscription"] = encode_subscription(msg.uid);
    /// XXX inefficient.
    result["payload"] = payload;
    result["topic"] = msg.topic;
    result["transit_id"] = msg.transit_id;
    result["event_ts"] = msg.event_ts;
    result["local_id"] = msg.local_id;
    result["content-encoding"] = msg.data["content-encoding"];
    result["dh"] = msg.data["dh"];
    result["salt"] = msg.data["salt"];

    json_value params;
    params["data"] = result;
    client_stream->push_request(params);

    log_data["websocket_stream"] = client_stream->ctx()->uniq_id();
    log_data["api_result"] = "notify success";

    // TODO implement acks with 200, 205 and so on
    // change state to @wait ack@ and dont accept notifications
    // (with same local_id or drop all anyway?)
    // Stream is held by catalogue and will wait for ack until incoming
    // connection is not closed. So we need to store notify request stream
    // and free it on timeout. We can store weak ptr or release notify stream
    // from expirable_stream callback.
    stream->result(http_codes::ok, "");
}

}}}
