#pragma once

#include "encoding.h"
#include "common.h"

namespace yxiva { namespace web { namespace webpushapi {

void subscribe(
    settings_ptr settings,
    const std::shared_ptr<websocket_rpc::stream>& stream,
    const string& subset,
    const string& public_key,
    const string& uidset_uuid)
{
    auto&& log_data = stream->ctx()->custom_log_data;
    log_data["api_group"] = "webpushapi subscriptions";

    if (subset.size() > settings->webpushapi.limits.public_key)
    {
        return send_bad_webpushapi_request(stream, "subscribe", "subset length limit exceeded");
    }
    log_data["subset"] = subset;

    if (public_key.size() > settings->webpushapi.limits.public_key)
    {
        return send_bad_webpushapi_request(stream, "subscribe", "public_key length limit exceeded");
    }

    if (uidset_uuid.size() > settings->webpushapi.limits.public_key)
    {
        return send_bad_webpushapi_request(
            stream, "subscribe", "uidset_uuid length limit exceeded");
    }
    log_data["uidset_uuid"] = uidset_uuid;

    auto uidsetid =
        subset.size() ? decode_uidsetid_from_subset(subset) : generate_uidsetid(stream->ctx());
    log_data["uidsetid"] = uidsetid;

    auto handler = handle_hub_list_size(
        stream,
        [stream, settings, uidsetid, uidset_uuid, public_key, &log_data](size_t size) mutable {
            if (size >= settings->webpushapi.max_subset_size)
            {
                uidsetid = generate_uidsetid(stream->ctx());
            }
            string uid = generate_uid(uidsetid);
            log_data["uid"] = uid;
            auto ttl = settings->webpushapi.subscription_ttl;
            find_hubrpc()->async_get(
                stream->ctx(),
                uidsetid,
                "/uidset/subscribe",
                { { "uidset", uidsetid },
                  { "uidset_uuid", uidset_uuid }, // TODO remove
                  { "uidset_extra", "" },         // TODO remove
                  { "uid", uid },
                  { "service", settings->webpushapi.service },
                  { "callback", settings->webpushapi.callback },
                  { "extra", "" },
                  { "filter", "" },
                  { "client", uidset_uuid },
                  { "session_key", public_key },
                  { "ttl", ttl },
                  { "id", settings->webpushapi.subscription_id } },
                [stream, settings, uidsetid, uid, ttl, &log_data](
                    const boost::system::error_code& ec, yhttp::response response) mutable {
                    if (ec || response.status != 200)
                    {
                        log_data["api_result"] = "subscribe failed";
                        log_data["error"] = "subscribe " + format_http_error(ec, response.status);
                        handle_default_hub_codes(stream, ec, response);
                        return;
                    }
                    auto subscription = encode_subscription(uid);
                    auto subscription_set = encode_subset(uidsetid);
                    log_data["subscription"] = subscription;
                    log_data["subscription_set"] = subscription_set;
                    log_data["api_result"] = "subscribed";
                    json_value result;
                    result["subscription"] = subscription;
                    result["subscription_set"] = subscription_set;
                    result["push_resource"] =
                        encode_push_resource(uid, settings->webpushapi.push_resource_url);
                    result["ttl"] = ttl * 1000;
                    stream->result_json(http_codes::ok, result);
                });
        });

    find_hubrpc()->async_get(
        stream->ctx(),
        uidsetid,
        "/uidset/list",
        { { "uidset", uidsetid }, { "service", settings->webpushapi.service } },
        [handler = std::move(handler), stream, subset, &log_data](auto& ec, auto response) mutable {
            if (ec || response.status != 200)
            {
                log_data["api_result"] = "subscribe failed";
                log_data["error"] = "list " + format_http_error(ec, response.status);
            }
            handler(ec, std::move(response));
        });
}

}}}
