#pragma once

#include "dispatcher.h"
#include "registry.h"
#include "stats.h"
#include <ymod_webserver/methods/transform.h>

namespace yxiva { namespace web { namespace websocket_rpc {

inline string path_extractor(std::shared_ptr<stream> s)
{
    return s->request()->url.make_full_path();
}

class main_handler : public yplatform::log::contains_logger
{
public:
    main_handler(
        settings_ptr st,
        std::shared_ptr<request_stats> root_stats,
        const yplatform::log::source& logger,
        const yplatform::log::tskv_logger& typed_logger)
        : yplatform::log::contains_logger(logger)
        , settings_(st)
        , session_registry_(new session_registry)
        , dispatcher_(new dispatcher(path_extractor))
        , stats_(new stats(root_stats))
        , typed_logger_(typed_logger)
    {
    }

    // Bindings must be specified before applications is started.
    template <typename Handler>
    void bind(const std::vector<string>& paths, Handler&& handler)
    {
        for (auto& path : paths)
        {
            (*dispatcher_)[path] = std::forward<Handler>(handler);
        }
    }
    template <typename Handler, typename Transformer>
    void bind(const std::vector<string>& paths, Handler&& handler, Transformer&& transformer)
    {
        for (auto& path : paths)
        {
            (*dispatcher_)[path] = ymod_webserver::make_transforming_proxy(
                std::forward<Transformer>(transformer), std::forward<Handler>(handler));
        }
    }

    const stats& get_stats() const
    {
        return *stats_;
    }

    void handle(websocket_stream_ptr stream)
    {
        if (stop_)
        {
            return;
        }
        if (settings_->websocket_rpc.tls_only && !stream->is_secure())
        {
            ymod_webserver::default_answers::send_forbidden(stream, "insecure connection");
            return;
        }
        auto sess = session_registry_->create(
            stream, settings_, dispatcher_, stats_, logger(), typed_logger_);
        sess->run();
    }

    // Need to call to shutdown() explicitly to ensure server stop.
    void shutdown()
    {
        stop_ = true;
        session_registry_->clear();
    }

private:
    settings_ptr settings_;
    bool stop_ = false;
    std::shared_ptr<session_registry> session_registry_;
    std::shared_ptr<dispatcher> dispatcher_;
    std::shared_ptr<stats> stats_;
    yplatform::log::tskv_logger typed_logger_;
};

}}}
