#pragma once

#include "dispatcher.h"
#include "stream.h"
#include "stats.h"
#include "../settings.h"
#include <yxiva/core/methods/check.h>
#include <ymod_webserver/codes.h>
#include <ymod_webserver/request.h>
#include <ymod_webserver/websocket.h>
#include <ymod_webserver/methods/dispatcher.h>

namespace yxiva { namespace web { namespace websocket_rpc {

// Session spawns streams and stores them to support close notifications.
// Access to the active streams container is synchronized via a strand.
// Session stores websocket stream as a weak pointer because
// it binds itself to websocket stream close events as a shared pointer.
// Websocket stream is held by itself while it is performing I/O operations.
class session
    : public std::enable_shared_from_this<session>
    , public yplatform::log::contains_logger
{
    struct stream_deleter
    {
        void operator()(stream* stream)
        {
            try
            {
                if (auto session = weak_session.lock())
                {
                    session->strand_.post([session, ctx = stream->ctx(), stream]() {
                        session->streams_.erase(ctx->uniq_id());
                        --session->stats_->active_streams;
                        try
                        {
                            delete stream;
                        }
                        catch (...)
                        {
                        }
                    });
                }
                else
                {
                    delete stream;
                }
            }
            catch (...)
            {
            }
        }
        std::weak_ptr<session> weak_session;
    };

public:
    session(
        const websocket_stream_ptr& ws_stream,
        settings_ptr settings,
        std::shared_ptr<dispatcher> dispatcher,
        std::shared_ptr<stats> stats,
        const yplatform::log::source& logger,
        const yplatform::log::tskv_logger& typed_logger);

    ~session();

    void run();
    void shutdown();

private:
    void shutdown_nosync();
    operation::result make_stream(
        const websocket_stream_ptr& stream,
        const json_value& json_msg,
        std::shared_ptr<websocket_rpc::stream>& rpc_stream);
    void execute(const string& json_rpc_request);
    bool message_limit_exceeded();
    bool rate_limit_exceeded(const std::shared_ptr<websocket_rpc::stream>& rpc_stream);
    bool route_incoming_message(const string& stream_id, const json_value& json_msg);

    boost::asio::io_service::strand strand_;
    yplatform::task_context_ptr ctx_;
    websocket_stream_weak_ptr ws_stream_;
    settings_ptr settings_;
    std::map<string, std::weak_ptr<stream>> streams_;
    string input_buffer_;
    size_t messages_received_ = 0;
    std::shared_ptr<dispatcher> dispatcher_;
    std::shared_ptr<stats> stats_;
    yplatform::log::tskv_logger typed_logger_;
};

}}}
