#include "stream.h"
#include <boost/lexical_cast.hpp>

namespace yxiva { namespace web { namespace websocket_rpc {

namespace ll = yplatform::log::typed;

// TODO extract WS RPC to webserver and use its' log tools.
static string replace_column_name_chars(const string& name)
{
    const char SPECIAL = '_';
    string ret;
    ret.reserve(name.size());
    bool beginning = true;
    bool escaped = false;
    for (auto& ch : name)
    {
        // Special char and digits in the beginning are denied.
        if (beginning && (!isalpha(ch))) continue;
        beginning = false;
        if (isalnum(ch))
        {
            ret += ch;
            escaped = false;
            continue;
        }
        if (!escaped)
        {
            ret += SPECIAL;
            // Column name can't contain special symbol sequences.
            escaped = true;
        }
    }
    return ret;
}

static void log_custom_data(const ymod_webserver::context& ctx, ll::attributes_map& os)
{
    for (auto& pair : ctx.custom_log_data)
    {
        auto key = replace_column_name_chars(pair.first);
        if (key.size()) os << ll::make_attr(key, pair.second);
    }
}

static void log_profilers_data(const ymod_webserver::context& ctx, ll::attributes_map& os)
{
    for (auto& v : ctx.profilers.get_values())
    {
        os << ll::make_attr(
            "profiler_" + replace_column_name_chars(v.name),
            ymod_webserver::prof_time_to_string(v.time));
    }
    os << ll::make_attr(
        "profiler_total", ymod_webserver::prof_time_to_string(ctx.profilers.get_total()));
}

static void write_to_access_log(
    const yplatform::log::tskv_logger& logger,
    const ymod_webserver::context_ptr& ctx,
    const ymod_webserver::context_ptr& session_ctx,
    const string& rpc_request_id,
    int code,
    const string& reason,
    ymod_webserver::request_ptr& req)
{
    ll::attributes_map am;
    log_custom_data(*ctx, am);
    log_profilers_data(*ctx, am);
    YLOG(logger, info) << ll::make_attr("event", "request")
                       << ll::make_attr("context", ctx->uniq_id())
                       << ll::make_attr("session", session_ctx->uniq_id())
                       << ll::make_attr("rpc_request_id", rpc_request_id)
                       << ll::make_attr("status_code", code) << ll::make_attr("reason", reason)
                       << ll::make_attr("request", req->url.make_full_path())
                       << ll::make_attr("remote_address", ctx->remote_address)
                       << ll::make_attr("remote_port", ctx->remote_port) << am;
}

stream::stream(
    websocket_stream_ptr stream,
    const string& request_id,
    request_ptr req,
    const yplatform::log::source& logger,
    const yplatform::log::tskv_logger& typed_logger)
    : yplatform::log::contains_logger(logger)
    , impl_(stream)
    , request_id_(request_id)
    , req_(req)
    , typed_logger_(typed_logger)
    , closed_{ false }
{
    ctx()->profilers.push("exec");
}

stream::~stream()
{
    try
    {
        if (!response_code_)
        {
            ctx()->profilers.pop("exec");
            write_to_access_log(
                typed_logger_, ctx(), impl_->ctx(), request_id_, 0, "no answer", req_);
        }
    }
    catch (...)
    {
    }
}

void stream::result(ymod_webserver::codes::code code, const string& reason)
{
    if (response_code_)
    {
        YLOG_CTX_LOCAL(ctx(), warning) << "detected response duplicate with code " << code;
    }
    else
    {
        send_response(*impl_, request_id_, code, reason);
        response_code_ = code;
        ctx()->profilers.pop("exec");
        write_to_access_log(
            typed_logger_, ctx(), impl_->ctx(), request_id_, code, code == 200 ? "" : reason, req_);
    }
}

void stream::result_json(ymod_webserver::codes::code code, const json_value& reason)
{
    if (response_code_)
    {
        YLOG_CTX_LOCAL(ctx(), warning) << "detected response duplicate with code " << code;
    }
    else
    {
        send_response(*impl_, request_id_, code, reason);
        response_code_ = code;
        ctx()->profilers.pop("exec");
        write_to_access_log(
            typed_logger_,
            ctx(),
            impl_->ctx(),
            request_id_,
            code,
            code == 200 ? "" : reason.stringify(),
            req_);
    }
}

void stream::push_request(const json_value& params)
{
    json_value message;
    message["method"] = "/push";
    message["params"] = params;
    message["id"] = ctx()->uniq_id();
    impl_->send_text(message.stringify());
}

void stream::push_notification(const json_value& params)
{
    json_value message;
    message["method"] = "/push";
    message["params"] = params;
    impl_->send_text(message.stringify());
}

void stream::pick_incoming_message(const json_value& data)
{
    if (message_hook_)
    {
        message_hook_(data);
    }
}

}}}
