#pragma once

#include "types.h"
#include "../settings.h"
#include <atomic>

namespace yxiva { namespace web { namespace websocket_rpc {

class stream : public yplatform::log::contains_logger
{
public:
    stream(
        websocket_stream_ptr stream,
        const string& request_id,
        request_ptr req,
        const yplatform::log::source& logger,
        const yplatform::log::tskv_logger& typed_logger);

    ~stream();

    context_ptr ctx() const
    {
        return req_->context;
    }

    request_ptr request() const
    {
        return req_;
    }

    bool is_secure() const
    {
        return impl_->is_secure();
    }

    void set_message_hook(const message_hook_t& hook)
    {
        message_hook_ = hook;
    }

    void set_close_hook(const close_hook_t& hook)
    {
        close_hook_ = hook;
    }

    void close(uint16_t code, const string& reason)
    {
        if (closed_.exchange(true)) return;

        if (close_hook_)
        {
            try
            {
                close_hook_(code, reason);
            }
            catch (const std::exception& e)
            {
                YLOG_CTX_LOCAL(ctx(), error) << "close hook exception: " << e.what();
            }
        }
    }

    void result(ymod_webserver::codes::code code, const string& reason = string());
    void result_json(ymod_webserver::codes::code code, const json_value& reason);
    void push_request(const json_value& params);
    void push_notification(const json_value& params);
    void pick_incoming_message(const json_value& data);

    auto bin_stream(size_t sz)
    {
        return impl_->bin_stream(sz);
    }

    yplatform::time_traits::timer_ptr make_timer() const
    {
        return impl_->make_timer();
    }

    boost::asio::io_service& get_io_service()
    {
        return impl_->get_io_service();
    }

private:
    websocket_stream_ptr impl_; // TODO store weak pointer?
    string request_id_;
    request_ptr req_;
    message_hook_t message_hook_;
    close_hook_t close_hook_;
    boost::optional<ymod_webserver::codes::code> response_code_;
    yplatform::log::tskv_logger typed_logger_;
    std::atomic<bool> closed_;
};

using stream_ptr = std::shared_ptr<stream>;

namespace detail {
inline json_value convert_request_id(const string& source)
{
    json_value result;
    if (source.size())
    {
        uint64_t numeric_id;
        if (boost::conversion::try_lexical_convert(source, numeric_id))
        {
            result = numeric_id;
        }
        else
        {
            result = source;
        }
    }
    return result;
}
}

template <typename Reason>
inline void send_response(
    websocket_stream& stream,
    const string& request_id,
    ymod_webserver::codes::code code,
    Reason&& reason)
{
    json_value answer;
    answer["id"] = detail::convert_request_id(request_id);
    if (code == 200)
    {
        answer["error"] = json_value();
        answer["result"] = std::forward<Reason>(reason);
    }
    else
    {
        answer["error"] = static_cast<int>(code);
        answer["reason"] = reason;
    }
    stream.send_text(answer.stringify());
}

}}}

#include "stream_helpers.ipp"
