#include "websocket_rpc_subscriber.h"
#include "mod_log/find.h"
#include "parse_push_ack.h"

#include <yxiva/core/split.h>

#include <boost/bind/protect.hpp>
#include <boost/pointer_cast.hpp>
#include <yplatform/find.h>
#include <processor/interface.h>

namespace yxiva { namespace web {

static const string NAME = string("ss_websocket_rpc");

websocket_rpc_subscriber::websocket_rpc_subscriber(
    task_context_ptr context,
    stream_ptr stream,
    formatters::formatter_ptr formatter,
    std::time_t sign_expiration,
    const string& client_id)
    : base_web_subscriber(context, formatter, sign_expiration, client_id), stream_(stream)
{
}

websocket_rpc_subscriber::~websocket_rpc_subscriber()
{
    close();
    reset_stream();
}

void websocket_rpc_subscriber::init()
{
    auto weak_self = boost::weak_ptr<base_web_subscriber>(shared_from_this());
    stream_->set_message_hook([this, weak_self](const json_value& value) {
        if (auto alive = weak_self.lock()) on_message(value);
    });
}

void websocket_rpc_subscriber::impl_notify_text(const string& message)
{
    impl_notify(message);
}

void websocket_rpc_subscriber::impl_notify_binary(const string& header, const string& message)
{
    stream_ptr stream_copy = stream();
    if (!stream_copy) return;
    auto bin_stream = stream_copy->bin_stream(header.size() + message.size());
    bin_stream << header << message;
}

void websocket_rpc_subscriber::ping(const ping_message& /*message*/)
{
    // nothing to do: no per stream pings in rpc connection
}

void websocket_rpc_subscriber::on_message(const json_value& value)
{
    push_ack ack;
    if (auto parsed = parse_push_ack(value, ack))
    {
    }
    else
    {
        YLOG_CTX_GLOBAL(ctx(), info) << parsed.error_reason;
        return;
    }
    find_xivaws_log()->log_notification_ack(
        ctx(), ack.send_context, ack.transit_id, ack.client, ack.service);
}

void websocket_rpc_subscriber::notify_error(const error_message& message)
{
    impl_notify(message);
}

void websocket_rpc_subscriber::close()
{
    stream_ptr stream_copy = stream();
    if (!stream_copy) return;
    stream_copy->close(websocket_codes::close_opcode_normal, "");
}

void websocket_rpc_subscriber::notify_position(const position_message& message)
{
    impl_notify(message);
}

void websocket_rpc_subscriber::notify_disconnected(const disconnected_message& message)
{
    impl_notify(message);
}

const string& websocket_rpc_subscriber::name() const
{
    return NAME;
}

websocket_rpc_subscriber::stream_ptr websocket_rpc_subscriber::stream()
{
    scoped_lock lock(mux_);
    return stream_;
}

void websocket_rpc_subscriber::reset_stream()
{
    scoped_lock lock(mux_);
    if (stream_)
    {
        stream_.reset();
    }
}

}}
